/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
  \file growisofs_pipe_regexp.c

   This file implements second part of cdw interface to growisofs.

   Code from growisofs_interface.c prepares commands strings to be executed by
   shell, sends them to run_command() and creates (only creates) processwin.

   Code in this file process output from growisofs (both from stdout and
   stderr) and calls processwin functions to display data in processwin.

   Pipe regexp code in this file is growisofs-specific only.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "cdw_task.h"
#include "cdw_processwin.h"
#include "gettext.h"
#include "cdw_regex_dispatch.h"
#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */
#include "cdw_growisofs_regex.h"
#include "cdw_debug.h"



/* note on burning with growisofs:

   when growisofs burns data writes files to DVD directly from hdd, user can
   see on console two different types of information:
   1. "89.37% done, estimate finish Fri Apr 25 22:03:43 2008"
   2. "10223616/118970368 ( 8.6%) @2.0x, remaining 0:42 "

   First type is output of mkisofs, which is used by growisofs to create ISO
   file system that will be written to DVD. This is information about progress
   made by mkisofs and it does not correspond directly to process of burning
   data to DVD. It may happen that initially mkisofs will produce some amount
   of bytes, and when internal buffer of growisofs will be full, mkisofs will
   have to wait until growisofs will start unloading content of the buffer
   onto DVD. Thus this information is not very trustworthy.

   Second type of output comes from growisofs itself, and it is (usually)
   information about amount of data actually burned to disc. Thus this is the
   information that we want to capture, process and display.

   growisofs has one command argument that "silences" mkisofs, not allowing
   information about mkisofs progress to be displayed. The option is
   "-use-the-force-luke=moi", and is used by cdw.

   You have to remember that when burning data files from hdd to DVD with
   growisofs you still have to call functions capturing output of mkisofs,
   because mkisofs may print to console some error or debugging information,
   which should be captured and presented to the user. An example of such
   information is error message about too long file names or about cyclic
   symlinks. So in pipe_regexp.c, when dispatching calls, always think about
   whether mkisofs will be used in background, as a helper tool, and call
   pipe regexp setup / execution / cleanup functions for mkisofs.

   There was handle_growisofs_mkisofs_image() function defined in this file,
   but since cdw always uses "-use-the-force-luke=moi" option, the function
   became unused, and was moved to unused_code.see. */

/* about recognizing situations, when there is no space left on optical
   disc for burning (growisofs 7.1):

   1. when you plan to burn files (either to empty disc or to disc which
   already carries some file system, but is not closed yet), but there is
   not enough space for the files, growisofs will print these two lines:

   :-( /dev/scd0: 2297888 blocks are free, 3282832 to be written!
   :-( write failed: No space left on device

   2. when you plan to burn ISO image to empty disc, but the file's size
   is larger than disc's capacity, growisofs will print this one line:

   :-( /dev/scd0: 2297888 blocks are free, 3282839 to be written!


   Since this module has access to thread_task (and its "id" field),
   assignment of appropriate error ID will be made only after detecting
   line about blocks, based on thread_task->id value. "No space left on
   device" line will not be detected and acted upon. */



extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];
extern cdw_task_t *thread_task;


/* Functions that response to given strings printed out by growisofs to
   stdout or stderr. It seems that only information about writing data to
   disc by growisofs is printed by growisofs to stdout - rest of messages
   is printed to stderr */

/* Handlers for messages printed before burning */
static void handle_growisofs_preformatting_blank_media(void);
static void handle_growisofs_restarting_format(void);

/* Handler for messages displayed during burning, showing progress
   of burning (for both data files and ISO image) */
static void cdw_growisofs_pr_handle_writing_data(regex_t *regex, regmatch_t *matches);

/* Handlers for various messages printed after burning
   (in various combinations) */
static void handle_growisofs_stopping_deicing(void);
static void handle_growisofs_flushing_cache(void);
static void handle_growisofs_updating_rma(void);
static void handle_growisofs_writing_leadout(void);
static void handle_growisofs_closing_track(void);
static void handle_growisofs_closing_session(void);
static void handle_growisofs_closing_disc(void);

/* Handlers for various problem or error messages */
static void handle_growisofs_not_recognized(void);
static void cdw_growisofs_pr_handle_no_space_left(regex_t *regex, regmatch_t *matches);
static void handle_growisofs_media_not_appendable(void);
static void handle_growisofs_use_z_option(void);
static void handle_growisofs_input_output_error(void);

/* Handlers for other messages */
static void handle_growisofs_version(regex_t *regex, regmatch_t *matches);


static regex_t regex1, regex2;
static regmatch_t *matches1 = NULL, *matches2 = NULL;

static regex_t eregex1, eregex2, eregex3, eregex4;
static regex_t /* eregex5*/  eregex6, eregex7, eregex8;
static regex_t eregex9, eregex10, eregex11 /*, eregex12 */;
static regex_t eregex13, eregex14, eregex15, eregex16, eregex17, eregex18;

static regmatch_t *ematches1 = NULL, *ematches2 = NULL, *ematches3 = NULL, *ematches4 = NULL;
static regmatch_t /* *ematches5 = NULL,*/ *ematches6 = NULL, *ematches7 = NULL, *ematches8 = NULL;
static regmatch_t *ematches9 = NULL, *ematches10 = NULL, *ematches11 = NULL, *ematches12 = NULL;
static regmatch_t *ematches13 = NULL, *ematches14 = NULL, *ematches15 = NULL, *ematches16 = NULL;
static regmatch_t *ematches17 = NULL, *ematches18 = NULL;

/*
 * Unfortunately growisofs doesn't put '100% finished" to stdout and
 * we cannot copy it to processwin. We have to simulate it in
 * handle_growisofs_flushing_cache() when growisofs puts "Flushing cache" to
 * its stderr. This variable let us be sure that writing was performed
 * and we can safely update ETA to 0s and progressbar to 100%.
 * The variable should be set to false when compiling regexps and set to true
 * in handle_growisofs_write_image() or handle_growisofs_write_files().
 */
static bool growisofs_writing_performed;

/* growisofs puts information about written data in following form:
 * " 1349943296/1389715456 ( 0.0%) @0x, remaining ??:?? ..."
 * first value is amount of data already written to disc (including
 * previous sessions), second value is amount of data that will be on disc
 * when current session will be written. I use these two values captured at
 * the beginning of process to calculate total size of data to be written
 * (second value - first value) - this will be calculated_total. Then I
 * calculate calculated done like this:
 * calculated_done = calculated_total - (second value - first value)
 *
 * calculated_total is reset in code preparing regexps and set to
 * (second value - first value) when growisofs_writing_performed is still false.
 *
 * currently unused, because growisofs not always prints the information as desired.
 */
/* static long calculated_total; */



/* regular expressions setup code */
void growisofs_stdout_regexp_prepare(void)
{
	/* resetting some values */
	growisofs_writing_performed = false;

	int rv;

	/* " 1349943296/1389715456 ( 0.0%) @0x, remaining ??:?? RBU 100.0% UBU   0.0%" */
	/* "   10223616/118970368 ( 8.6%) @2.0x, remaining 0:42 RBU 100.0% UBU   3.1%" */
	rv = regcomp(&regex1, "([0-9]*)/([0-9]+)([ ]*)([(])([ ]*)([0-9?]+).([0-9?]+)%([)])([ ]*)@([0-9]+)([.]*)([0-9]*)x,([ ]*)remaining([ ]+)([0-9?]+):([0-9?]+)", REG_EXTENDED);
	matches1 = (regmatch_t *) calloc(1, (regex1.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex1, 101);
	}

	/* "acerion@macondo3 5 $ growisofs --version " */
	/* "* growisofs by <appro@fy.chalmers.se>, version 7.1, " */
	rv = regcomp(&regex2, "growisofs by <appro@fy.chalmers.se>, version ([0-9]+).([0-9]+)", REG_EXTENDED);
	matches2 = (regmatch_t *) calloc(1, (regex2.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex2, 102);
	}

	return;
}





void growisofs_stderr_regexp_prepare(void)
{
	int rv;
	growisofs_writing_performed = false;

	/* "/dev/scd0: pre-formatting blank DVD+RW..." */
	rv = regcomp(&eregex1, "pre-formatting blank", REG_EXTENDED);
	ematches1 = (regmatch_t *) calloc(1, (eregex1.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex1, 201);
	}

	/* "writing lead-out" */
	rv = regcomp(&eregex2, "writing lead-out", REG_EXTENDED);
	ematches2 = (regmatch_t *) calloc(1, (eregex2.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex2, 202);
	}

	/* "restarting DVD+RW format" */
	rv = regcomp(&eregex3, "restarting DVD[+]RW format", REG_EXTENDED);
	ematches3 = (regmatch_t *) calloc(1, (eregex3.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex3, 203);
	}

	/* "media is not recognized as recordable DVD" */
	rv = regcomp(&eregex4, "media is not recognized as recordable DVD", REG_EXTENDED);
	ematches4 = (regmatch_t *) calloc(1, (eregex4.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex4, 204);
	}

	/* growisofs after direct writing: "/dev/scd0: flushing cache" */
	rv = regcomp(&eregex6, "flushing cache", REG_EXTENDED);
	ematches6 = (regmatch_t *) calloc(1, (eregex6.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex6, 206);
	}

	/* growisofs after flushing cache: "/dev/scd0: closing track" */
	rv = regcomp(&eregex7, "closing track", REG_EXTENDED);
	ematches7 = (regmatch_t *) calloc(1, (eregex7.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex7, 207);
	}

	/* growisofs after closing track: "/dev/scd0: closing session" */
	rv = regcomp(&eregex8, "closing session", REG_EXTENDED);
	ematches8 = (regmatch_t *) calloc(1, (eregex8.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex8, 208);
	}

	/* growisofs after direct writing single session and flushing disc: "/dev/scd0: updating RMA" */
	rv = regcomp(&eregex9, "updating RMA", REG_EXTENDED);
	ematches9 = (regmatch_t *) calloc(1, (eregex9.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex9, 209);
	}

	/* growisofs after writing single session, flushing cache and updating RMA: "/dev/scd0: closing disc" */
	rv = regcomp(&eregex10, "closing disc", REG_EXTENDED);
	ematches10 = (regmatch_t *) calloc(1, (eregex10.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex10, 210);
	}

	/* growisofs after writing data (image only??): "/dev/scd0: stopping de-icing" */
	rv = regcomp(&eregex11, "stopping de-icing", REG_EXTENDED);
	ematches11 = (regmatch_t *) calloc(1, (eregex11.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex11, 211);
	}

#if 0   /* disabled - see note on top of this file about detecting when there is not enough
	   space for data */
	/* write failed: No space left on device */
	/* this message is printed when size of selected files is larger than
	   size of free space on optical disc size  */
	rv = regcomp(&eregex12, "No space left on device", REG_EXTENDED);
	ematches12 = (regmatch_t *) calloc(1, (eregex12.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex12, 212);
	}
#endif

	/* :-( media is not appendable */
	rv = regcomp(&eregex13, "media is not appendable", REG_EXTENDED);
	ematches13 = (regmatch_t *) calloc(1, (eregex13.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex13, 213);
	}

	/* you most likely want to use -Z option */
	/* printed when -M option is used for disc without valid filesystem */
	rv = regcomp(&eregex14, "you most likely want to use -Z option", REG_EXTENDED);
	ematches14 = (regmatch_t *) calloc(1, (eregex14.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex14, 214);
	}

	/* /dev/scd0 doesn't look like isofs... */
	/* printed when -M option is used for disc without valid filesystem */
	rv = regcomp(&eregex15, "doesn't look like isofs", REG_EXTENDED);
	ematches15 = (regmatch_t *) calloc(1, (eregex15.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex15, 215);
	}

	/* write failed: Input/output error */
	/* yet another error, I don't know why it happens */
	rv = regcomp(&eregex16, "write failed: Input/output error", REG_EXTENDED);
	ematches16 = (regmatch_t *) calloc(1, (eregex16.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex16, 216);
	}

	/* ":-( /dev/scd0: 2297888 blocks are free, 2687932 to be written!" */
	/* this message is printed when size of ISO image file is larger than
	   optical disc size; note that there are two separate messages about
	   lack of sufficient space: one for burning files, and one for burning
	   image */
	rv = regcomp(&eregex17, ":-[(] /dev/scd0: ([0-9]+) blocks are free, ([0-9]+) to be written!", REG_EXTENDED);
	ematches17 = (regmatch_t *) calloc(1, (eregex17.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex17, 217);
	}

	/* next session would cross 4GB boundary, aborting */
	/* printed when file larger than 4GB was selected for burning */
	rv = regcomp(&eregex18, "next session would cross 4GB boundary, aborting", REG_EXTENDED);
	ematches18 = (regmatch_t *) calloc(1, (eregex18.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex18, 218);
	}

	return;
}





void growisofs_stdout_regexp_execute(void)
{
	cdw_sdm ("called, stdout_pipe_buffer = %s\n", stdout_pipe_buffer);

	int rv;
	rv = regexec(&regex1, stdout_pipe_buffer, regex1.re_nsub + 1, matches1, 0);
	if (rv == 0) {
		cdw_sdm ("calling cdw_growisofs_pr_handle_writing_data() (stdout)\n");

		cdw_growisofs_pr_handle_writing_data(&regex1, matches1);
	}

	rv = regexec(&regex2, stdout_pipe_buffer, regex2.re_nsub + 1, matches2, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_version() (stdout)\n");

		handle_growisofs_version(&regex2, matches2);
	}

	return;
}





void growisofs_stderr_regexp_execute(void)
{
	cdw_sdm ("stderr_pipe_buffer = %s\n", stderr_pipe_buffer);

	int rv = 0;
	/* preformatting blank media. */
	rv = regexec(&eregex1, stderr_pipe_buffer, eregex1.re_nsub + 1, ematches1, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_preformatting_blank_media() (stderr)\n");
		handle_growisofs_preformatting_blank_media();
	}

	/* writing lead-out. */
	rv = regexec(&eregex2, stderr_pipe_buffer, eregex2.re_nsub + 1, ematches2, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_writing_leadout() (stderr)\n");
		handle_growisofs_writing_leadout();
	}

	rv = regexec(&eregex3, stderr_pipe_buffer, eregex3.re_nsub + 1, ematches3, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_restarting_format() (stderr)\n");
		handle_growisofs_restarting_format();
	}

	/* "media is not recognized as recordable DVD" */
	rv = regexec(&eregex4, stderr_pipe_buffer, eregex4.re_nsub + 1, ematches4, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_not_recognized() (stderr)\n");
		handle_growisofs_not_recognized();
	}

	/* growisofs after direct writing: "/dev/scd0: flushing cache" */
	rv = regexec(&eregex6, stderr_pipe_buffer, eregex6.re_nsub + 1, ematches6, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_flushing_cache() (stderr)\n");
		handle_growisofs_flushing_cache();
	}

	/* growisofs after flushing_cache: "/dev/scd0: closing track" */
	rv = regexec(&eregex7, stderr_pipe_buffer, eregex7.re_nsub + 1, ematches7, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_closing_track() (stderr)\n");
		handle_growisofs_closing_track();
	}

	/* growisofs after closing track: "/dev/scd0: closing session" */
	rv = regexec(&eregex8, stderr_pipe_buffer, eregex8.re_nsub + 1, ematches8, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_closing_session() (stderr)\n");
		handle_growisofs_closing_session();
	}

	/* growisofs after flushing_cache: "/dev/scd0: updating RMA" */
	rv = regexec(&eregex9, stderr_pipe_buffer, eregex9.re_nsub + 1, ematches9, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_closing_track() (stderr)\n");
		handle_growisofs_updating_rma();
	}

	/* growisofs after closing track: "/dev/scd0: closing disc" */
	rv = regexec(&eregex10, stderr_pipe_buffer, eregex10.re_nsub + 1, ematches10, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_closing_session() (stderr)\n");
		handle_growisofs_closing_disc();
	}

	/* growisofs after writing data (image only??): "/dev/scd0: stopping de-icing" */
	rv = regexec(&eregex11, stderr_pipe_buffer, eregex11.re_nsub + 1, ematches11, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_stopping_deicing() (stderr)\n");
		handle_growisofs_stopping_deicing();
	}

#if 0   /* disabled - see note on top of this file about detecting when there is not enough
	   space for data */
	/* growisofs when attempting to write too much data: "write failed: No space left on device" */
	rv = regexec(&eregex12, stderr_pipe_buffer, eregex12.re_nsub + 1, ematches12, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_no_space_left() (stderr)\n");
		handle_growisofs_no_space_left();
	}
#endif

	/* :-( media is not appendable */
	rv = regexec(&eregex13, stderr_pipe_buffer, eregex13.re_nsub + 1, ematches13, 0);
	if (rv == 0) {
		cdw_sdm ("calling handle_growisofs_media_not_appendable() (stderr)\n");
		handle_growisofs_media_not_appendable();
	}

	/* "you most likely want to use -Z option" */
	/* it means that growisofs was called in "append" mode for empty disc;
	   this shouldn't happen anymore because cdw now can recognize if disc
	   is empty or not, and adjust available growisofs options accordingly */
	rv = regexec(&eregex14, stderr_pipe_buffer, eregex14.re_nsub + 1, ematches14, 0);
	if (rv == 0) {
		cdw_vdm ("calling handle_growisofs_use_z_option()\n");
		handle_growisofs_use_z_option();
	}

	/* /dev/scd0 doesn't look like isofs... */
	rv = regexec(&eregex15, stderr_pipe_buffer, eregex15.re_nsub + 1, ematches15, 0);
	if (rv == 0) {
		cdw_vdm ("calling handle_growisofs_use_z_option()\n");
		handle_growisofs_use_z_option();
	}

	/* write failed: Input/output error */
	rv = regexec(&eregex16, stderr_pipe_buffer, eregex16.re_nsub + 1, ematches16, 0);
	if (rv == 0) {
		cdw_vdm ("calling handle_growisofs_input_output_error()\n");
		handle_growisofs_input_output_error();
	}

	/* ":-( /dev/scd0: 2297888 blocks are free, 2687932 to be written!" */
	rv = regexec(&eregex17, stderr_pipe_buffer, eregex17.re_nsub + 1, ematches17, 0);
	if (rv == 0) {
		cdw_vdm ("calling cdw_growisofs_pr_handle_no_space_left()\n");
		cdw_growisofs_pr_handle_no_space_left(&eregex17, ematches17);
	}

	/* "next session would cross 4GB boundary, aborting" */
	rv = regexec(&eregex18, stderr_pipe_buffer, eregex18.re_nsub + 1, ematches18, 0);
	if (rv == 0) {
		thread_task->tool_status.growisofs |= CDW_TOOL_STATUS_GROWISOFS_4GB_BOUNDARY;
		cdw_vdm ("ERROR: tool_status.growisofs |= CDW_TOOL_STATUS_GROWISOFS_4GB_BOUNDARY\n");
	}

	return;
}





void growisofs_stdout_regexp_destroy(void)
{
	regfree(&regex1);
	free(matches1);
	matches1 = (regmatch_t *) NULL;

	regfree(&regex2);
	free(matches2);
	matches2 = (regmatch_t *) NULL;

	return;
}





void growisofs_stderr_regexp_destroy(void)
{
	regfree(&eregex1);
	regfree(&eregex2);
	regfree(&eregex3);
	regfree(&eregex4);
	/* regfree(&eregex5); */
	regfree(&eregex6);
	regfree(&eregex7);
	regfree(&eregex8);
	regfree(&eregex9);
	regfree(&eregex10);
	regfree(&eregex11);
	/* regfree(&eregex12); */
	regfree(&eregex13);
	regfree(&eregex14);
	regfree(&eregex15);
	regfree(&eregex16);
	regfree(&eregex17);
	regfree(&eregex18);

	free(ematches1);
	free(ematches2);
	free(ematches3);
	free(ematches4);
	/* free(ematches5); */
	free(ematches6);
	free(ematches7);
	free(ematches8);
	free(ematches9);
	free(ematches10);
	free(ematches11);
	/* free(ematches12); */
	free(ematches13);
	free(ematches14);
	free(ematches15);
	free(ematches16);
	free(ematches17);
	free(ematches18);

	ematches1 = (regmatch_t *) NULL;
	ematches2 = (regmatch_t *) NULL;
	ematches3 = (regmatch_t *) NULL;
	ematches4 = (regmatch_t *) NULL;
	/* ematches5 = (regmatch_t *) NULL; */
	ematches6 = (regmatch_t *) NULL;
	ematches7 = (regmatch_t *) NULL;
	ematches8 = (regmatch_t *) NULL;
	ematches9 = (regmatch_t *) NULL;
	ematches10 = (regmatch_t *) NULL;
	ematches11 = (regmatch_t *) NULL;
	ematches12 = (regmatch_t *) NULL;
	ematches13 = (regmatch_t *) NULL;
	ematches14 = (regmatch_t *) NULL;
	ematches15 = (regmatch_t *) NULL;
	ematches16 = (regmatch_t *) NULL;
	ematches17 = (regmatch_t *) NULL;
	ematches18 = (regmatch_t *) NULL;

	return;
}





/**
   \brief Process output of growisofs when burning files or image to DVD

   The function will be called when writing actual data (not lead-out or
   lead-in or session info), that is when:
    \li writing files directly from hdd to dvd
    \li when writing iso image file to dvd
    \li when blanking DVD (by writing from /dev/zero device)

   growisofs behaves similarly in these cases, but the function shows less
   information in processwin when blanking is performed.

   You should also read comment at top of the file ("note on burning with growisofs")
*/
void cdw_growisofs_pr_handle_writing_data(regex_t *regex, regmatch_t *matches)
{
	/* This is a regexp for burning with growisofs
              1        2      3     4    5      6         7        8    9      10      11    12        13            14      15        16
	  "([0-9]*)/([0-9]+)([ ]*)([(])([ ]*)([0-9?]+).([0-9?]+)%([)])([ ]*)@([0-9]+)([.]*)([0-9]*)x,([ ]*)remaining([ ]+)([0-9?]+):([0-9?]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 16);

	int perc_decimal = 0, perc_fract = 0;
	int speed_decimal = 0, speed_fract = 0;
	int minutes = 0, seconds = 0;

	for (unsigned int i = 6; i <= regex->re_nsub; i++) {
		char submatch[PIPE_BUFFER_SIZE];
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

		if (i == 6) { /* percentage - decimal part */
			if (!strcmp(submatch, "??")) {
				perc_decimal = 0;
			} else {
				perc_decimal = atoi(submatch);
			}
			cdw_sdm ("INFO: submatch %d, decimal part of percentage: \"%s\" -> %d\n", i, submatch, perc_decimal);
		} else if (i == 7) { /* percentage - fractional part */
			if (!strcmp(submatch, "??")) {
				perc_fract = 0;
			} else {
				perc_fract = atoi(submatch);
			}
			cdw_sdm ("INFO: submatch %d, fractional part of percentage: \"%s\" -> %d\n", i, submatch, perc_fract);
		} else if (i == 10) { /* speed - decimal part */
			if (!strcmp(submatch, "??")) {
				speed_decimal = 0;
			} else {
				speed_decimal = atoi(submatch);
			}
			cdw_vdm ("INFO: submatch %d, decimal part of speed: \"%s\" -> %d\n", i, submatch, speed_decimal);
		} else if (i == 12) { /* speed - fractional part */
			if (!strcmp(submatch, "??")) {
				speed_fract = 0;
			} else {
				speed_fract = atoi(submatch);
			}
			cdw_vdm ("INFO: submatch %d, fractional part of speed: \"%s\" -> %d\n", i, submatch, speed_fract);
		} else if (i == 15) { /* minutes remaining */
			if (!strcmp(submatch, "??")) {
				minutes = 99;
			} else {
				minutes = atoi(submatch);
			}
			cdw_sdm ("INFO: submatch %d, minutes remaining: \"%s\" -> %d\n", i, submatch, minutes);
		} else if (i == 16) { /* seconds remaining */
			if (!strcmp(submatch, "??")) {
				seconds = 99;
			} else {
				seconds = atoi(submatch);
			}
			cdw_sdm ("INFO: submatch %d, seconds remaining: \"%s\" -> %d\n", i, submatch, seconds);
		} else {
			;
		}
	}

	growisofs_writing_performed = true;

	double percent = perc_decimal + 0.1 * perc_fract;
	double total = 0.0;
	double done = 0.0;
	char progress_string[PROCESSWIN_MAX_RTEXT_LEN + 1];

	if (thread_task->id == CDW_TASK_ERASE_DISC) {
		total = 100.0;
		done = percent;
		/* there is no valid information about megabytes burned to
		   disc during erasing, so this string must be empty */
		progress_string[0] = '\0';
		cdw_vdm ("INFO: progress_string = \"%s\" (empty)\n", progress_string);
	} else { /* burn from image or burn from files */

		/* during tests files I will use files larger than 1MB */
		cdw_assert (thread_task->burn.data_size_mb > 1.0,
			    "ERROR: something is wrong with thread_task->data_size_mb, its value is small: %f\n",
			    thread_task->burn.data_size_mb);

		/* works for both burning iso file and burning files from disc */
		total = thread_task->burn.data_size_mb;
		done = (float) ((total * percent) / 100.0);

		/* 2TRANS: this is string displayed in process progress window;
		   first %.1f is amount of data already written, second %.1f is
		   total amount of data to be written */
		snprintf(progress_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("%.1f/%.1f MB"), done, total);
		cdw_sdm ("INFO: progress_string = \"%s\"\n", progress_string);
	}

	cdw_vdm ("INFO: done / total = %f / %f (task: %s)\n", done, total,
		 thread_task->id == CDW_TASK_BURN_FROM_IMAGE ? "burn from image" :
		 (thread_task->id == CDW_TASK_BURN_FROM_FILES ? "burn from files" : "erase"));

	if (thread_task->id == CDW_TASK_BURN_FROM_IMAGE) {
		/* 2TRANS: this is string displayed in process progress window;
		   writing iso image to DVD disc is in progress */
		cdw_processwin_display_sub_info(_("Writing image in progress..."));
	} else if (thread_task->id == CDW_TASK_BURN_FROM_FILES) {
		/* 2TRANS: this is string displayed in process progress window;
		   writing selected files to DVD disc is in progress */
		cdw_processwin_display_sub_info(_("Writing files in progress..."));
	} else if (thread_task->id == CDW_TASK_ERASE_DISC) {
		/* 2TRANS: this is string displayed in process progress window;
		   blanking DVD disc is in progress */
		cdw_processwin_display_sub_info(_("Erasing of DVD in progress..."));
	} else {
		cdw_processwin_display_sub_info((char *) NULL);
	}

	cdw_processwin_display_progress_conditional((int) (done), (int) (total),
						    progress_string);

	char eta_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is message displayed in process progress window:
	   ETA stands for time left to end of process, %2d is amount of
	   minutes left, %02d is amount of seconds left */
	sprintf(eta_string, _("ETA: %2d:%02d"), minutes, seconds);
	cdw_vdm ("INFO: ETA: minutes = %d, seconds = %d -> string = \"%s\"\n", minutes, seconds, eta_string);
	cdw_processwin_display_eta(eta_string);

	char speed_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is message displayed in process progress window;
	   "Speed" is writing speed, %d and %d are decimal and fractional
	   parts of writing speed, e.g. "Speed: 2.5x" */
	sprintf(speed_string, _("Speed: %d.%dx"), speed_decimal, speed_fract);
	cdw_vdm ("INFO: speed: decimal = %d, fract = %d -> string = \"%s\"\n",
		 speed_decimal, speed_fract, speed_string);
	cdw_processwin_display_fifo_and_speed(-1, speed_decimal, speed_fract);

	cdw_processwin_wrefresh();

	return;
}





void handle_growisofs_version(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing growisofs version is
                                                            1         2
	   "growisofs by <appro@fy.chalmers.se>, version ([0-9]+)\.([0-9]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 2);

	int major = 0, minor = 0;
	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		char submatch[PIPE_BUFFER_SIZE];
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

		if (i == 0) { /* whole string, do nothing */
			cdw_vdm ("INFO: submatch %d, whole string =\"%s\"\n", i, submatch);
		} else if (i == 1) { /* major version number */
			cdw_vdm ("INFO: submatch %d, major version number = \"%s\"\n", i, submatch);
			major = atoi((char *) &submatch);
		} else if (i == 2) { /* major version number */
			cdw_vdm ("INFO: submatch %d, minor version number = \"%s\"\n", i, submatch);
			minor = atoi((char *) &submatch);
		} else {
			;
		}
	}

	cdw_vdm ("INFO: growisofs version is %d.%d\n", major, minor);

	return;
}





void handle_growisofs_preformatting_blank_media(void)
{
	/* 2TRANS: this is message displayed in process progress window:
	   dvd+rw disc in drive has never been written to and has to be
	   formatted for a first time */
	cdw_processwin_display_sub_info(_("Pre-formatting blank disc..."));
	cdw_processwin_wrefresh();

	return;
}





void handle_growisofs_writing_leadout(void)
{
	/* 2TRANS: this is message displayed in process progress window:
	   writing is almost finished, growisofs is writing ending part
	   of session */
	cdw_processwin_display_sub_info(_("Writing leadout..."));
	cdw_processwin_wrefresh();

	return;
}





void handle_growisofs_restarting_format(void)
{
	/* 2TRANS: this is message displayed in process progress window */
	cdw_processwin_display_sub_info(_("Reformatting media..."));
	cdw_processwin_wrefresh();

	return;
}





void handle_growisofs_not_recognized(void)
{
	/* 2TRANS: this is message displayed in process progress window */
	cdw_processwin_display_sub_info(_("Wrong media type, cannot write"));
	cdw_processwin_wrefresh();

	sleep(3);
	return;
}





/**
   After writing actual data to disc growisofs flushes cache.
   This function displays information about this in processwin.
*/
void handle_growisofs_flushing_cache(void)
{
	cdw_sdm ("called\n");

	/* 2TRANS: this is message displayed in processwin:
	   writing is almost finished, growisofs is cleaning cache buffer */
	cdw_processwin_display_sub_info(_("Flushing cache..."));

	/* Flushing cache means that writing to disc is finished.
	   Unfortunately growisofs doesn't put "100% finished" to stdout and
	   we cannot display such information in processwin. We have to
	   simulate it here. Just check that writing was in fact  performed. */
	if (growisofs_writing_performed) {
		char eta_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
		/* 2TRANS: this is message displayed in process progress window:
		   ETA stands for amount of time left to end of current process;
		   0:00 is minutes:seconds, please leave value as zero */
		snprintf(eta_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("ETA: 0:00"));

		cdw_processwin_display_eta(eta_string);
		cdw_processwin_display_progress(100, 100, (char *) NULL);
	}

	cdw_processwin_wrefresh();

	return;
}





/**
   After writing actual data to disc and flushing cache growisofs closes track.
   This function displays information about this in processwin.
*/
void handle_growisofs_closing_track(void)
{
	cdw_sdm ("called\n");

	/* 2TRANS: this is message displayed in processwin:
	   writing is finished, growisofs is closing track */
	cdw_processwin_display_sub_info(_("Closing track..."));

	cdw_processwin_wrefresh();

	return;
}





/**
   After writing actual data to disc and closing track, growisofs closes
   session. This function displays information about this in processwin.
*/
void handle_growisofs_closing_session(void)
{
	cdw_sdm ("called\n");

	/* 2TRANS: this is message displayed in processwin:
	   writing is finished, growisofs is closing session */
	cdw_processwin_display_sub_info(_("Closing session..."));
	cdw_processwin_wrefresh();

	return;
}





void handle_growisofs_updating_rma(void)
{
	cdw_sdm ("called\n");

	/* 2TRANS: this is message displayed in processwin:
	   writing single session is finished, growisofs flushed cache and
	   now is doing something called "Updating RMA", whatever it is ;) */
	cdw_processwin_display_sub_info(_("Updating RMA..."));
	cdw_processwin_wrefresh();

	return;
}





void handle_growisofs_closing_disc(void)
{
	cdw_sdm ("called\n");

	/* 2TRANS: this is message displayed in processwin:
	   writing single session is finished, growisofs flushed cache and
	   now is closing disc, so that no other data can be written  */
	cdw_processwin_display_sub_info(_("Closing disc..."));
	cdw_processwin_wrefresh();

	return;
}





void handle_growisofs_stopping_deicing(void)
{
	cdw_sdm ("called\n");

	/* 2TRANS: this is message displayed in processwin;
	   I have no idea what it means */
	cdw_processwin_display_sub_info(_("Stopping de-icing..."));
	cdw_processwin_wrefresh();

	return;
}




#if 0   /* disabled - see note on top of this file about detecting when
	   there is not enough space for data */
/* write failed: No space left on device */
void handle_growisofs_no_space_left(void)
{
	cdw_sdm ("called\n");

	/* 2TRANS: this is message displayed in processwin;
	   An error occurred during process of writing */
	cdw_processwin_display_text_info(1, _("An error occurred"));
	/* 2TRANS: this is message in displayed in processwin;
	   growisofs detected that there is not enough space on disc
	   to write selected data */
	cdw_processwin_display_sub_info(_("Too little space on optical disc"));
	cdw_processwin_wrefresh();
	cdw_processwin_wgetch();

	thread_task->tool_status |= CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_LEFT;
	cdw_vdm ("ERROR: tool_status |= CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_LEFT\n");

	return;
}
#endif




/* :-( media is not appendable */
/* this should be detected by higher level code and in practice this function
   will never be called, but just in case... */
void handle_growisofs_media_not_appendable(void)
{
	cdw_sdm ("called\n");

	/* 2TRANS: this is message displayed in processwin;
	   An error occurred during process of writing */
	cdw_processwin_display_main_info(_("An error occurred"));
	/* 2TRANS: this is message in displayed in processwin;
	   growisofs detected that there is not enough space on disc
	   to write selected data */
	cdw_processwin_display_sub_info(_("Media is not appendable"));
	cdw_processwin_wrefresh();
	cdw_processwin_wgetch();

	thread_task->tool_status.growisofs |= CDW_TOOL_STATUS_GROWISOFS_MEDIA_NOT_APPENDABLE;
	cdw_vdm ("ERROR: tool_status.growisofs |= CDW_TOOL_STATUS_GROWISOFS_MEDIA_NOT_APPENDABLE\n");

	return;
}





/* in case of DVD-RW Restricted and DVD+RW media cdw sometimes can't tell
   if media is empty and allows user to select "append to disc" option
   instead presenting him only with "start new disc" option;
   result is that growisofs doesn't burn data to disc and exits with
   "you most likely want to use -Z option" message */
void handle_growisofs_use_z_option(void)
{
	cdw_sdm ("called\n");

	/* 2TRANS: this is message displayed in processwin */
	cdw_processwin_display_main_info(_("An error occurred"));
	/* 2TRANS: this is message in displayed in processwin;
	   growisofs detected that there is not enough space on disc
	   to write selected data */
	cdw_processwin_display_sub_info(_("Use different session writing mode"));
	cdw_processwin_wrefresh();
	cdw_processwin_wgetch();

	thread_task->tool_status.growisofs |= CDW_TOOL_STATUS_GROWISOFS_USE_Z_OPTION;
	cdw_vdm ("ERROR: tool_status.growisofs |= CDW_TOOL_STATUS_GROWISOFS_USE_Z_OPTION\n");

	return;
}



/* "write failed: Input/output error" */
void handle_growisofs_input_output_error(void)
{
	cdw_sdm ("called\n");

	/* 2TRANS: this is message displayed in processwin */
	cdw_processwin_display_main_info(_("An error occurred"));
	/* 2TRANS: this is message in displayed in processwin;
	   growisofs detected that there is not enough space on disc
	   to write selected data */
	cdw_processwin_display_sub_info(_("Some error occurred, please try again with different settings"));
	cdw_processwin_wrefresh();
	cdw_processwin_wgetch();

	thread_task->tool_status.growisofs |= CDW_TOOL_STATUS_GROWISOFS_IO_ERROR;
	cdw_vdm ("ERROR: tool_status.growisofs |= CDW_TOOL_STATUS_GROWISOFS_IO_ERROR\n");

	return;
}





void cdw_growisofs_pr_handle_no_space_left(regex_t *regex, regmatch_t *matches)
{
	/* This is a regexp for catching message when there is no space
	   left on optical disc for data
	                       1                          2
	   ":-[(] /dev/scd0: ([0-9]+) blocks are free, ([0-9]+) to be written!" */
	cdw_regex_assert_subex_number(regex->re_nsub, 2);

	/* if this is the case, then value of 1 is smaller than value of 2
	   we should check this below just to be sure */

	long int blocks_free = 0;
	long int blocks_to_be_written = 0;

	char submatch[PIPE_BUFFER_SIZE];
	int len = 0;
	unsigned int sub = 1;
	len = cdw_regex_get_submatch(matches, sub, stderr_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return;
	} else {
		blocks_free = atol(submatch);
		cdw_vdm ("INFO: submatch %d, blocks free: \"%s\" -> %ld\n", sub, submatch, blocks_free);
	}

	sub = 2;
	len = cdw_regex_get_submatch(matches, sub, stderr_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return;
	} else {
		blocks_to_be_written = atol(submatch);
		cdw_vdm ("INFO: submatch %d, blocks to be written: \"%s\" -> %ld\n", sub, submatch, blocks_to_be_written);
	}

	if (blocks_free < blocks_to_be_written) {
		if (thread_task->id == CDW_TASK_BURN_FROM_IMAGE) {
			cdw_vdm ("WARNING: not enough space on optical disc for ISO image file\n");
			thread_task->tool_status.growisofs |= CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_IMAGE;
		} else if (thread_task->id == CDW_TASK_BURN_FROM_FILES) {
			cdw_vdm ("WARNING: not enough space on optical disc for data files\n");
			thread_task->tool_status.growisofs |= CDW_TOOL_STATUS_GROWISOFS_NO_SPACE_FOR_FILES;
		}
	} else {
		cdw_vdm ("ERROR: %s() triggered, but \"blocks_free < blocks_to_be_written\" condition not met\n", __func__);
	}

	return;
}
