/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
   \file cdw_fs_browser.c

   File implements a widget that allows browsing in native file system.
   It can be used to select group of files to be burned to CD, or to
   select single file, e.g. ISO9660 image file.
*/

#define _BSD_SOURCE /* strdup() */
#define _GNU_SOURCE /* strndup() */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "cdw_fs_browser.h"
#include "gettext.h"
#include "cdw_file.h"
#include "cdw_fs.h"
#include "cdw_file_picker.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_debug.h"
#include "canonicalize.h"
#include "cdw_config.h"

extern cdw_config_t global_config;


static cdw_fs_browser_t   *cdw_fs_browser_new_base(WINDOW *window);
static unsigned long long  cdw_fs_browser_read_dir_content(cdw_dll_item_t **list, const char *dirpath);
static void                cdw_fs_browser_internal_error_dialog(int rv);
static cdw_rv_t            cdw_fs_browser_change_dir_on_enter(cdw_fs_browser_t *browser);


#ifndef NDEBUG
#define cdw_fs_browser_debug_report_current_fullpath(macro_browser, macro_message) \
	{								\
		char *macro_f = cdw_fs_browser_get_current_fullpath(macro_browser); \
		cdw_vdm ("INFO: %s, current fullpath = \"%s\"\n", macro_message, macro_f); \
		free(macro_f);						\
		macro_f = (char *) NULL;				\
	}
#else
#define cdw_fs_browser_debug_report_current_fullpath(macro_browser, macro_message)
#endif



/**
   \brief Create new file system browser

   Create new fs browser widget. Use \p window as a main window for the
   browser: browser will use \p window as its main window. Set initial
   location in file system to \p initial_fullpath

   \param window - initialized ncurses window to be used as main browser window
   \param initial_fullpath - initial location in file system

   \return pointer to correctly initialized file system browser on success
   \return null pointer on failure
*/
cdw_fs_browser_t *cdw_fs_browser_new(WINDOW *window, const char *initial_fullpath)
{
	cdw_fs_browser_t *browser = cdw_fs_browser_new_base(window);
	if (browser == (cdw_fs_browser_t *) NULL) {
		cdw_vdm ("ERROR: failed to create browser's base\n");
		return (cdw_fs_browser_t *) NULL;
	}

	/* make several attempts to browse to initial location */

	/* 1. first try to browse to location specified by caller */
	int rv = cdw_fs_browser_browse_to(browser, initial_fullpath);
	if (rv == 0) {
		/* success, browsed to given location and refreshed the view */
		return browser;
	}

	/* 2. try to browse to a dir that is one level up from location specified by caller */
	ssize_t i = cdw_fs_get_filename_start(initial_fullpath);
	if (i == -1) {
		cdw_vdm ("WARNING: failed to get file name start in \"%s\"\n", initial_fullpath);
	} else {
		char *parent = strndup(initial_fullpath, (size_t) i);
		cdw_assert (parent != (char *) NULL, "ERROR: failed to strndup fullpath \"%s\"\n", initial_fullpath);
		rv = cdw_fs_browser_browse_to(browser, parent);
		free(parent);
		parent = (char *) NULL;
		if (rv == 0) {
			/* success, browsed to given 'parent' location and refreshed the view */
			return browser;
		}
	}

	/* 3. try to browse to "safe" location provided by fs module.
	   cdw_fs_get_initial_dirpath() tries to return one of following:
	   current working dir, "home" dir, "/tmp" dir, or "/" (root) dir */
	char *tmp = cdw_fs_get_initial_dirpath();
	if (tmp != (char *) NULL) {
		rv = cdw_fs_browser_browse_to(browser, tmp);
		if (rv == 0) {
			/* success, browsed to given location and refreshed the view */
			return browser;
		} else {
			cdw_vdm ("ERROR: can't browse to any safe initial dirpath for fs browser, this is bad\n");
		}
	} else {
		cdw_vdm ("ERROR: can't get any safe initial dirpath for fs browser, this is bad\n");
	}

	cdw_fs_browser_internal_error_dialog(CDW_ERROR);
	cdw_fs_browser_delete(&browser);
	return (cdw_fs_browser_t *) NULL;
}





/**
   \brief Wrapper for some initial code creating file system browser

   Function that hides some boring details of creating new file system
   browser like allocating memory for new fs browser data structure,
   setting widget size, initializing empty full paths etc.

   \param window - non-NULL parent window, in which a fs browser will be displayed

   \return pointer to new, partially initialized fs browser on success
   \return NULL on problems
*/
cdw_fs_browser_t *cdw_fs_browser_new_base(WINDOW *window)
{
	cdw_assert (window != (WINDOW *) NULL, "ERROR: \"window\" argument is NULL\n");

	cdw_fs_browser_t *browser = (cdw_fs_browser_t *) malloc (sizeof (cdw_fs_browser_t));
	if (browser == (cdw_fs_browser_t *) NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for fs browser\n");
		return (cdw_fs_browser_t *) NULL;
	}

	int n_lines = getmaxy(window);
	cdw_assert (n_lines > 2, "ERROR: number of lines of window is too small: %d\n", n_lines);
	n_lines -= 2;
	int n_cols = getmaxx(window);
	cdw_assert (n_cols > 2, "ERROR: number of cols of window is too small: %d\n", n_cols);
	n_cols -= 2;

	/* 1, 1 - begin_y, begin_x */
	browser->display = cdw_list_display_new(window, n_lines, n_cols, 1, 1, CDW_COLORS_DIALOG);
	if (browser->display == (CDW_LIST_DISPLAY *) NULL) {
		cdw_vdm ("ERROR: failed to create file_browser display\n");
		cdw_fs_browser_internal_error_dialog(CDW_ERROR);

		free(browser);
		browser = (cdw_fs_browser_t *) NULL;

		return (cdw_fs_browser_t *) NULL;
	}

	browser->display->display_item = cdw_file_display_file;
	/* space=select, enter=change dir, q = quit, dot=toggle displaying hidden files, 0=guard */
	cdw_list_display_add_return_keys(browser->display, KEY_DC, ' ', CDW_KEY_ENTER, 'q', 'Q', '.', KEY_BACKSPACE, '~', '?', 0);
#ifndef NDEBUG
	cdw_list_display_add_return_keys(browser->display,
					 KEY_DOWN, KEY_UP, KEY_NPAGE, KEY_PPAGE, KEY_HOME, KEY_END, 0);
#endif

	return browser;
}





/**
   \brief Deallocate all resources related to given browser widget

   Function frees all resources related to given \p browser (including
   browser itself) or calls functions that do their job to free
   some resources. \p browser is set to NULL on return, but it can't be
   NULL (nor it can point to NULL) on entry.

   \param browser - browser to be freed/destroyed
*/
void cdw_fs_browser_delete(cdw_fs_browser_t **browser)
{
	cdw_assert (browser != (cdw_fs_browser_t **) NULL, "ERROR: you passed null pointer\n");

	if (*browser == (cdw_fs_browser_t *) NULL) {
		cdw_vdm ("WARNING: you passed pointer to null browser\n");
		return;
	}

	cdw_file_dealloc_files_from_list((*browser)->display->list);

	cdw_list_display_delete(&((*browser)->display));
	(* browser)->display = (CDW_LIST_DISPLAY *) NULL;

	free(*browser);
	*browser = (cdw_fs_browser_t *) NULL;

	return;
}





/**
   \brief Read information about files in given directory, store the information on a list

   Function scans directory specified by \p dirpath, checks information about
   all files (files and directories)  in the directory, and saves the
   information on \p list - one node of 'cdw_file_t *' per file. Function
   returns number of items in \p list. Reference to current directory (".")
   is not put on the list.

   There will be at least one item in the \p list: reference to parent dir.
   If displaying of hidden files is disabled in configuration, the list won't
   put them on the \p list, and returned value won't include hidden files.
   Therefore the function returns number of items on the list - not number
   of items (files+dirs) in directory.

   In case of errors the \p list is cleaned, \p list is set to null, and
   function returns 0.

   You can pass \p list that already carries some information, but it will
   be modified: either set to contain new, correct data, or reset to empty
   list.

   \param list - list to be filled with information about files
   \param dirpath - path to directory which you want to scan

   \return number of files read from current dir into list
   \return 0 on errors
*/
unsigned long long cdw_fs_browser_read_dir_content(cdw_dll_item_t **list, const char *dirpath)
{
	if (*list != (cdw_dll_item_t *) NULL) {
		cdw_file_dealloc_files_from_list(*list);
		cdw_dll_clean(*list);
		*list = (cdw_dll_item_t *) NULL;
	}

	struct dirent **eps = (struct dirent **) NULL;
	/*  by passing one() to scandir() we make scandir() to omit "." item;
	    caution: scandir mallocs() space for eps, don't forget to free() it */
	ssize_t n_files = scandir(dirpath, &eps, cdw_scandir_filter_one, alphasort);

	if (n_files == -1) {
		/* this happened to me once when after suspending to RAM and
		   then bringing system up I ran cdw and browsed to secondary
		   disc - the disc wasn't reconnected (?) after coming from
		   suspension to RAM */
		cdw_vdm ("ERROR: scandir() returns -1 for dirpath \"%s\"\n", dirpath);
		cdw_fs_browser_internal_error_dialog(CDW_ERROR);
		return 0;
	} else if (n_files == 0) {
		/* not going to happen, with scandir() called with one(),
		   there will be at least '..' item in eps */
		cdw_assert (n_files != 0, "ERROR: scandir() called with one() returs 0 for dirpath \"%s\"\n", dirpath);;
		return 0;
	} else {
		unsigned long long n = (unsigned long long) n_files;
		unsigned long long cp = cdw_fs_copy_dirent_to_list(list, dirpath, eps, n, global_config.fs.display_hidden_files);

		/* eps is no longer needed, all interesting information about
		   directory content is now in display->list */
		for (unsigned long long i = 0; i < n; i++) {
			free(eps[i]);
			eps[i] = (struct dirent *) NULL;
		}
		free(eps);
		eps = (struct dirent **) NULL;

		if (cp == 0) {
			if (*list != (cdw_dll_item_t *) NULL) {
				cdw_file_dealloc_files_from_list(*list);
				cdw_dll_clean(*list);
				*list = (cdw_dll_item_t *) NULL;
			}
			cdw_vdm ("ERROR: failed to copy dirent to list for dirpath \"%s\"\n", dirpath);
			return 0;
		} else {
#ifndef NDEBUG
			cdw_vdm ("INFO: Content of current directory \"%s\":\n", dirpath);
			for (unsigned long long i = 0; i < cp; i++) {
				cdw_dll_item_t *item = cdw_dll_ith_item(*list, (size_t) i);
			        cdw_file_t *file = (cdw_file_t *) item->data;
				ssize_t f = cdw_fs_get_filename_start(file->fullpath);
				cdw_vdm ("INFO:       file #%6lld = \"%s\"\n", i, file->fullpath + f);
			}
#endif
			return cp;
		}
	}
}





/**
   \brief Visit new directory in given display

   Function should be called when user presses ENTER on any file in
   fs browser. The function checks if current item is a directory
   and if it is, then the function tries to enter the directory.
   If current item is not a dir then nothing happens, nothing is
   changed, and function returns CDW_NO.

   If function enters the directory then content of new directory is read
   to list associated with \p browser, the content is displayed by the
   browser's display, and browser's dirpath is updated to new current
   directory.

   If resolving new dir path fails because of an error, or new dir content
   cannot be read because of an error, browser's dirpath is not modified
   and list associated with browser's display is reset to empty list.

   \param browser - fs browser with associated display that should be updated with content of new dir

   \return CDW_OK if directory was successfully changed and displayed
   \return CDW_NO if directory was not changed, but not because of error
   \return CDW_ERROR if other error occurred and dir was not changed because of that error
*/
cdw_rv_t cdw_fs_browser_change_dir_on_enter(cdw_fs_browser_t *browser)
{
	cdw_assert (browser->display != (CDW_LIST_DISPLAY *) NULL, "ERROR: display is null\n");
	cdw_assert (browser->display->n_items > browser->display->current_item_ind,
		    "ERROR, item index (0-based) is %zd but number of items is %zd\n",
		    browser->display->current_item_ind, browser->display->n_items);

	cdw_file_t *file = cdw_fs_browser_get_current_file(browser);
	if (file->type == CDW_FS_FILE || file->type != CDW_FS_DIR || file->invalid) {
		/* ENTER on file has no meaning, on invalid file has no
		   result, nothing needs to be changed */
		return CDW_NO;
	} else if (file->is_ref_to_parent_dir && !strcmp(file->fullpath, "/../")) {
		/* we are in root dir, nowhere else to go up to */
		return CDW_OK;
	} else {
		;
	}

	int rv = 0;
	char *new_dirpath = (char *) NULL;
	if (file->is_ref_to_parent_dir) {
		/* "/some/dir/../", remove last dir from the path */
		new_dirpath = cdw_fs_shorten_fullpath(file);
		if (new_dirpath == (char *) NULL) {
			cdw_vdm ("ERROR: failed to shorten fullpath \"%s\"\n", file->fullpath);
			return CDW_ERROR;
		}

		/* browse_to_file() moves cursor (focus) to directory item from
		   which we have just ascended instead of going to 0-th item
		   in newly visited dir */
		rv = cdw_fs_browser_browse_to_file(browser, new_dirpath);
	} else {
		/* canonicalize because current item may be a link */
		new_dirpath = canonicalize_filename_mode(file->fullpath, CAN_MISSING);
		if (new_dirpath == (char *) NULL) {
			/* canonicalization of dirpath failed */
			cdw_vdm ("ERROR: failed to canonicalize path \"%s\"\n", file->fullpath);
			return CDW_ERROR;
		}
		rv = cdw_fs_browser_browse_into_dir(browser, new_dirpath);
	}

	if (rv != 0) {
		cdw_vdm ("ERROR: failed to browse to new dirpath \"%s\"\n", new_dirpath);

		if (rv == CDW_FS_CHECK_E_PERM_NO_X) {
			cdw_vdm ("ERROR: no \"exec\" permissions\n");
			cdw_fs_browser_internal_error_dialog(rv);

		} else if (rv == CDW_FS_CHECK_E_PERM_NO_R) {
			cdw_vdm ("ERROR: no \"read\" permissions\n");
			cdw_fs_browser_internal_error_dialog(rv);

		} else if (rv == CDW_FS_CHECK_SYS_ERROR) {
			cdw_vdm ("ERROR: system error\n");
			cdw_fs_browser_internal_error_dialog(rv);

		} else {
			cdw_vdm ("ERROR: reason unknown\n");
			cdw_fs_browser_internal_error_dialog(CDW_ERROR);
		}
		cdw_list_display_refresh(browser->display);
	}
	free(new_dirpath);
	new_dirpath = (char *) NULL;

	if (rv == 0) {
		return CDW_OK;
	} else if (rv == CDW_FS_CHECK_E_PERM_NO_X || rv == CDW_FS_CHECK_E_PERM_NO_R) {
		/* not a good reason to close fs browser window, but
		   a good one not to browse into dir */
		return CDW_NO;
	} else {
		return CDW_ERROR;
	}
}





/**
   \brief Function that controls display's driver

   \p browser has associated list display widget, and every list display
   widget has a driver. This function controls the driver by calling it
   in a loop and checking it's return values. The function reacts to
   some of the return values in a way that you would expect from file
   system browser: the function calls another function to change working
   directory (when user pressed ENTER in display).

   The function returns when:
   \li dir was successfully changed (so that owner of this fs browser
       can decide what to do next) - Enter key was pressed;
   \li user pressed SPACE key in list display
   \li user pressed ESCAPE key in list display
   \li error occurred

   The three keys (Enter, Space, Escape) were configured when fs browser
   was created.

   Caller of this function should check value of browser->display->item_i
   to see which file is the current one on function's return.

   \param browser - file system browser that should be controlled with this function

   \return CDW_KEY_ENTER if user pressed ENTER and fs browser successfully changed current working dir
   \return ' ' (SPACE) if user pressed space on some file
   \return CDW_KEY_ESCAPE if user pressed ESCAPE key in the widget
   \return -1 on errors
*/
int cdw_fs_browser_driver(cdw_fs_browser_t *browser)
{
	int key = 'a';
	while (key != CDW_KEY_ESCAPE) {
		key = cdw_list_display_driver(browser->display);
		if (key == CDW_KEY_ENTER) {
			cdw_rv_t crv = cdw_fs_browser_change_dir_on_enter(browser);
			if (crv == CDW_OK) {
				cdw_fs_browser_debug_report_current_fullpath(browser, "INFO: ENTER key pressed");
			} else if (crv == CDW_ERROR) {
				cdw_vdm ("ERROR: failed to change dir in fs browser\n");
				return -1;
			} else { /* CDW_NO - not a dir or no perms to change dir, continue */
				;
			}
		} else if (key == ' ') {
			cdw_fs_browser_debug_report_current_fullpath(browser, "INFO: SPACE key pressed");
#ifndef NDEBUG
			/* SPACE key may be used by higher level code for selecting
			   files (if SPACE is defined as "return on" char) */
			cdw_assert (browser->display->n_items > browser->display->current_item_ind,
				    "ERROR: index of current file is larger than number of files\n");
#endif
		} else if (key == '.') {
			if (global_config.fs.display_hidden_files) {
				global_config.fs.display_hidden_files = false;
			} else {
				global_config.fs.display_hidden_files = true;
			}
			char *f = cdw_fs_browser_get_current_fullpath(browser);
			/* FIXME: check return values */
			int rv = cdw_fs_browser_browse_to_file(browser, f);
			free(f);
			f = (char *) NULL;
			assert (!rv);
			key = 'a'; /* dummy value, keep looping */
		} else if (key == KEY_BACKSPACE) {
			cdw_file_t *file = cdw_fs_browser_get_current_file(browser);
			char *new_dir = cdw_fs_shorten_fullpath(file);
			if (new_dir != (char *) NULL) {
				cdw_fs_browser_browse_to_file(browser, new_dir);
				free(new_dir);
				new_dir = (char *) NULL;
			} else {
				cdw_vdm ("ERROR: can't go up from \"%s\"\n", file->fullpath);
			}
			return key; /* allow client code to update current path */
		} else if (key == '~') {
			const char *dir = cdw_fs_get_home_dir_fullpath();
			cdw_fs_browser_browse_into_dir(browser, dir);
			return key; /* allow client code to update current path */
		} else if (cdw_list_display_is_return_char(browser->display, key)) {
			/* some other return key that client code
			   has to act upon */
			return key;
		} else {
			;
		}
#ifndef NDEBUG
		if (key == KEY_DOWN
		    || key == KEY_UP
		    || key == KEY_NPAGE
		    || key == KEY_PPAGE
		    || key == KEY_HOME
		    || key == KEY_END) {

			cdw_fs_browser_debug_report_current_fullpath(browser, "INFO: movement key pressed");
		}
#endif

		if (cdw_list_display_is_return_char(browser->display, key)) {
			break;
		}
	}

	return key;
}





/**
   \brief Display dialog window with error message

   Function displays dialog window with error message. Function created to
   avoid repetition of the same dialog in two places.

   FIXME: the function accepts values of regular type int
   and of type cdw_rv_t, but type of its argument is int.
*/
void cdw_fs_browser_internal_error_dialog(int rv)
{
	char *message = (char *) NULL;
	if (rv == CDW_ERROR) {
		/* 2TRANS: this is message in dialog window */
		message = _("Unexpected error, perhaps some file has been deleted. Please close this window and open it again.");
	} else if (rv == CDW_FS_CHECK_E_PERM_NO_X) {
		/* 2TRANS: this is message in dialog window;
		   "change directory" means "enter into directory" */
		message = _("Cannot change directory: wrong permissions.");
	} else if (rv == CDW_FS_CHECK_E_PERM_NO_R) {
		/* 2TRANS: this is message in dialog window;
		   "read directory" means "read content of directory" */
		message = _("Cannot read directory: wrong permissions.");
	} else if (rv == CDW_FS_CHECK_SYS_ERROR) {
		/* 2TRANS: this is message in dialog window:
		   user highlighted a file and pressed enter,
		   but the file probably does not exit */
		message = _("File does not exists or cannot change directory.");
	} else if (rv == CDW_NO) {
		/* 2TRANS: this is message in dialog window. Some
		   function returned error value. No further action
		   will be performed, no other explanations provided */
		message = _("Cannot display content of current directory. Please close this window and open it again.");
	} else {
		return;
	}

	/* 2TRANS: this is title of dialog window */
	cdw_buttons_dialog(_("Error"), message,
			   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

	return;
}





cdw_file_t *cdw_fs_browser_get_current_file(cdw_fs_browser_t *browser)
{
	cdw_assert (browser != (cdw_fs_browser_t *) NULL, "ERROR: browser pointer is NULL\n");

	void *data = cdw_list_display_get_current_item_data(browser->display);
	if (data == (void *) NULL) {
		cdw_assert (data != (void *) NULL,
			    "ERROR: failed to fetch current file #%zd from fs browser->display\n",
			    browser->display->current_item_ind);
		return (cdw_file_t *) NULL;
	} else {
		cdw_file_t *file = (cdw_file_t *) data;
		cdw_vdm ("INFO: fetching file #%zd = \"%s\"\n", browser->display->current_item_ind, file->fullpath);
		return file;
	}
}





int cdw_fs_browser_browse_to(cdw_fs_browser_t *browser, const char *fullpath)
{
	struct stat finfo;
	if (stat(fullpath, &finfo) == -1) {
		cdw_vdm ("ERROR: failed to stat() \"%s\"\n", fullpath);
		return -1;
	}

	if (S_ISDIR(finfo.st_mode)) {
		int rv = cdw_fs_browser_browse_into_dir(browser, fullpath);
		if (rv == 0) {
			return 0;
		} else {
			cdw_vdm ("ERROR: failed to browse to new dir %s\n", fullpath);
			return -1;
		}
	} else if (S_ISREG(finfo.st_mode)) { /* file */
		int rv = cdw_fs_browser_browse_to_file(browser, fullpath);
		if (rv == 0) {
			return 0;
		} else if (rv == ENOENT) {
			cdw_vdm ("ERROR: no such dir: \"%s\"\n", fullpath);
			return ENOENT;
		} else {
			cdw_vdm ("ERROR: unknown error %d\n", rv);
			return -1;
		}
	} else {
		cdw_vdm ("INFO: neither dir nor file: %s\n", fullpath);
		return ENONET;
	}
}





int cdw_fs_browser_browse_into_dir(cdw_fs_browser_t *browser, const char *dirpath)
{
	int rv = cdw_fs_check_dirpath(dirpath);
	if (rv != 0) {
		/* not a dir, or can't visit dir or can't read dir's content */
		return rv;
	}

	/* n is a number of files on list, not number of files in
	   directory. The two may be different if displaying of hidden
	   files is turned off. */
	unsigned long long n = cdw_fs_browser_read_dir_content(&(browser->display->list), dirpath);
	if (n == 0) {
		cdw_assert (browser->display->list == (cdw_dll_item_t *) NULL,
			    "ERROR: dir not read properly, but list of files is not null\n");

		browser->display->n_items = 0;
		cdw_vdm ("ERROR: failed to read dir content for new dirpath \"%s\"\n", dirpath);

		return -1;
	} else {
#ifndef NDEBUG
		size_t len = cdw_dll_length(browser->display->list);
		cdw_assert (len == (size_t) n, "ERROR: lengths mismatch: dir content size (n) = %lld, len of browser display list (len) = %zd\n", n, len);
#endif
		browser->display->n_items = (size_t) n;
		/* first 0 - position cursor on first item in new dir
		   second 0 - horizontal offset, true - highlight selected item */
		cdw_list_display_scroll_to(browser->display, 0, 0, true);

		cdw_fs_browser_debug_report_current_fullpath(browser, "INFO: successfully read new dir");
		return 0;
	}
}





/**
   \return ENONET if there is no directory specified in fullpath
                  (or there are no access rights to the directory)
   \return ENOENT if there is a dir specified in fullpath, but there
                  is no file specified by fullpath
   \return 0 on success
*/
int cdw_fs_browser_browse_to_file(cdw_fs_browser_t *browser, const char *fullpath)
{
	ssize_t f = cdw_fs_get_filename_start(fullpath);
	cdw_assert (f >= 0, "ERROR: can't get start of file name in \"%s\"\n", fullpath);
	char *dirpath = strndup(fullpath, (size_t) f);
	cdw_assert (dirpath != (char *) NULL, "ERROR: failed to strndup() dirpath from \"%s\"\n", fullpath);

	int rv = cdw_fs_browser_browse_into_dir(browser, dirpath);
	if (rv != 0) {
		cdw_vdm ("ERROR: failed to browse to new dir \"%s\"\n", fullpath);
		free(dirpath);
		dirpath = (char *) NULL;
		return ENONET;
	} else {
		/* we are in directory specified by fullpath, now we need
		   to move cursor (highlight) to file in this directory,
		   specified by fullpath; this search may not always succeed,
		   e.g. because the file has been removed from native file
		   system, or never existed */
		bool found = false;
		size_t item_ind = 0;
		cdw_sdm ("INFO: ==== searching for file \"%s\"\n", fullpath + f);
		for (item_ind = 0; item_ind < browser->display->n_items; item_ind++) {

			/* TODO: optimize this line by traversing list directly */
			cdw_dll_item_t *list_item =  cdw_dll_ith_item(browser->display->list, item_ind);
			cdw_file_t *file_from_list = (cdw_file_t *) list_item->data;
			cdw_sdm ("INFO: ==== matching against file \"%s\"\n", file_from_list->fullpath + file_from_list->name_start);

			if (!strcmp(fullpath + f, file_from_list->fullpath + file_from_list->name_start)) {
				found = true;
				break;
			}
		}

		if (found) {
			cdw_list_display_scroll_to(browser->display, item_ind, 0, true);
			free(dirpath);
			dirpath = (char *) NULL;

			return 0;
		} else {
			free(dirpath);
			dirpath = (char *) NULL;

			/* file may be not available for one reason:
			   it is a hidden file, but displaying of files
			   has been turned off in the middle of browsing;
			   this is not an error, handle this gracefully */
			cdw_vdm ("WARNING: file \"%s\" not found in dir \"%s\"\n", fullpath + f, dirpath);
			ssize_t start = cdw_fs_get_filename_start(fullpath);
			if (cdw_fs_is_hidden(fullpath + start) && !global_config.fs.display_hidden_files) {
				cdw_vdm ("INFO: it may be because displaying hidden files has been disabled\n");
				cdw_list_display_scroll_to(browser->display, 0, 0, true);
				return 0;
			} else {
				return ENOENT;
			}
		}
	}
}





char *cdw_fs_browser_get_current_dirpath(cdw_fs_browser_t *browser)
{
	cdw_file_t *file = cdw_fs_browser_get_current_file(browser);
	cdw_assert (file != (cdw_file_t *) NULL, "ERROR: can't get current file\n");
	ssize_t i = cdw_fs_get_filename_start(file->fullpath);
	cdw_assert (i > 0, "ERROR: can't get file name start\n");
	char *dirpath = strndup(file->fullpath, (size_t) i);
	cdw_assert (dirpath != (char *) NULL, "ERROR: can't strdup dirpath\n");
	return dirpath;
}





char *cdw_fs_browser_get_current_printable_dirpath(cdw_fs_browser_t *browser)
{
	cdw_file_t *file = cdw_fs_browser_get_current_file(browser);
	cdw_assert (file != (cdw_file_t *) NULL, "ERROR: can't get current file\n");
	if (file->printable_fullpath != (char *) NULL) {
		ssize_t i = cdw_fs_get_filename_start(file->printable_fullpath);
		cdw_assert (i > 0, "ERROR: can't get file name start\n");
		char *dirpath = strndup(file->printable_fullpath, (size_t) i);
		cdw_assert (dirpath != (char *) NULL, "ERROR: can't strdup dirpath\n");
		return dirpath;
	} else {
		return cdw_fs_browser_get_current_dirpath(browser);
	}
}





char *cdw_fs_browser_get_current_fullpath(cdw_fs_browser_t *browser)
{
	cdw_file_t *file = cdw_fs_browser_get_current_file(browser);
	cdw_assert (file != (cdw_file_t *) NULL, "ERROR: can't get current file\n");
	cdw_assert (file->fullpath != (char *) NULL, "ERROR: current file has no fullpath\n");
	/* file->fullpath is a valid pointer at this point, but if
	   you will pass it to function that re-reads (and thus reallocates)
	   list with current dir listing, then the pointer becomes
	   invalid an you are in trouble; so let's just allocate
	   new string */
	char *fullpath = strdup(file->fullpath);
	return fullpath;
}





bool cdw_fs_browser_is_navigation_key(int key)
{
	if (key == KEY_DOWN
	    || key == KEY_UP
	    || key == KEY_NPAGE
	    || key == KEY_PPAGE
	    || key == KEY_HOME
	    || key == KEY_END) {

		return true;
	} else {
		return false;
	}
}





/**
   \brief Display window with information about available hotkeys

   Information is displayed in window that is no larger than
   given \p browser, and \p browser window is automatically refreshed
   after closing help window.

   \p browser - fs browser, on top of which to display a help window
*/
void cdw_fs_browser_help_window(cdw_fs_browser_t *browser)
{
	int n_lines = 10;
	int n_cols = browser->display->n_cols < 50 ? browser->display->n_cols : 50;
	int begin_y = browser->display->begin_y + (browser->display->n_lines / 2) - (n_lines / 2);
	int begin_x = browser->display->begin_x + (browser->display->n_cols / 2) - (n_cols / 2);

	WINDOW *window = cdw_ncurses_window_new((WINDOW *) NULL,
						n_lines, n_cols,
						begin_y, begin_x,
						CDW_COLORS_DIALOG, "Help", "Press any key to close");
	if (window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create new window\n");
		return;
	}
	mvwprintw(window, 1, 2, "Hot keys available in this window:");
	mvwprintw(window, 3, 2, " .        - toggle displaying hidden files");
	mvwprintw(window, 4, 2, " ~        - go to home directory");
	mvwprintw(window, 5, 2, "backspace - go to parent directory");
	mvwprintw(window, 6, 2, " ?        - display this help");

	wgetch(window);
	delwin(window);
	window = (WINDOW *) NULL;

	wrefresh(browser->display->subwindow);

	return;
}


