/////////////////////////////////////////////////////////////
//                                                         //
// Copyright (c) 2003-2011 by The University of Queensland //
// Earth Systems Science Computational Centre (ESSCC)      //
// http://www.uq.edu.au/esscc                              //
//                                                         //
// Primary Business: Brisbane, Queensland, Australia       //
// Licensed under the Open Software License version 3.0    //
// http://www.opensource.org/licenses/osl-3.0.php          //
//                                                         //
/////////////////////////////////////////////////////////////

//----------------------------------------------
//       CSoftBWallInteractionGroup functions
//----------------------------------------------

#include "Foundation/console.h"

template<class T>
CSoftBWallInteractionGroup<T>::CSoftBWallInteractionGroup(TML_Comm* comm):AWallInteractionGroup<T>(comm)
{}

/*!
  Constructor for bonded wall interaction group with direction dependend elasticity

  \param comm the communicator
  \param wallp a pointer to the wall
  \param param the interaction parameters
*/
template<class T>
CSoftBWallInteractionGroup<T>::CSoftBWallInteractionGroup(TML_Comm* comm,CWall* wallp, const CSoftBWallIGP* I)
  : AWallInteractionGroup<T>(comm)
{
  console.XDebug() << "making CSoftBWallInteractionGroup \n";

  m_normalK=I->getNormalK();
  m_shearK=I->getShearK();
  this->m_wall=wallp;
  m_tag=I->getTag();
  m_scaling=I->getScaling();
//  console.XDebug() << "kx, ky, kz: " << m_kx << ","<< m_ky << ","<< m_kz << "\n";
}

template<class T>
void CSoftBWallInteractionGroup<T>::calcForces()
{

  console.XDebug() << "calculating " << m_interactions.size() << " soft bonded wall forces\n" ;

  for(
    typename vector<CSoftBondedWallInteraction<T> >::iterator it=m_interactions.begin();
    it != m_interactions.end();
    it++
  ){
    it->calcForces();
  }
}

template<class T>
void CSoftBWallInteractionGroup<T>::Update(ParallelParticleArray<T>* PPA)
{

  console.XDebug() << "CSoftBWallInteractionGroup::Update()\n" ;

  // empty particle list first
  m_interactions.erase(m_interactions.begin(),m_interactions.end());
  // build new particle list
  typename ParallelParticleArray<T>::ParticleListHandle plh=
    PPA->getParticlesAtPlane(this->m_wall->getOrigin(),this->m_wall->getNormal());
  for(typename ParallelParticleArray<T>::ParticleListIterator iter=plh->begin();
      iter!=plh->end();
      iter++){
    if((*iter)->getTag()==m_tag){
      bool iflag=PPA->isInInner((*iter)->getPos());
      m_interactions.push_back(CSoftBondedWallInteraction<T>(*iter,this->m_wall,m_normalK,m_shearK,m_scaling,iflag));
    }
  }
  console.XDebug() << "end CSoftBWallInteractionGroup::Update()\n";
}

template<class T>
ostream& operator<<(ostream& ost,const CSoftBWallInteractionGroup<T>& IG)
{
  ost << "CBWallInteractionGroup" << endl << flush;
  ost << *(IG.m_wall) << endl << flush;
 
  return ost;
}
