
/* Copyright(c)'2007 by Gilles Villard
   Created Mon Oct 29 12:35:16 CET 2007
   See COPYING for license information
*/


#ifndef FPLLL_RATIONAL_CPP
#define FPLLL_RATIONAL_CPP

#include <gmp.h>
#include "rational.h"



#include <iostream>
#include <cstdlib>
#include <cmath>

#include <nr.h>

using namespace std;

// ---------------------------- Cstor and dstor 

Rational::Rational()
{
  mpq_init((mpq_ptr)&rep);
}


Rational::~Rational()
{
  mpq_clear((mpq_ptr)&rep);
}



// -------------- Initialization and assignment 
 

inline void Rational::set(const Rational& q)
{
  mpq_set(rep, (mpq_ptr)&q.rep);
}

inline void Rational::set(const signed long int n, const signed long int d)
{
  mpq_set_si((mpq_ptr)&rep, n, d);
  mpq_canonicalize((mpq_ptr)&rep);
}

inline void Rational::set_ui(const unsigned long int n, const unsigned long int d)
{
  mpq_set_ui((mpq_ptr)&rep, n, d);
  mpq_canonicalize((mpq_ptr)&rep);
}

inline void Rational::set(const mpz_t n)
{
  mpq_set_z((mpq_ptr)&rep, n);
}

inline void Rational::set(Z_NR<mpz_t> n)
{
  mpq_set_z((mpq_ptr)&rep, n.GetData());
}

inline void Rational::set(Z_NR<long int> n)
{
  mpq_set_si((mpq_ptr)&rep, n.GetData(), 1);
}

// -------------- Comparing rationals 

inline int Rational::cmp(const Rational& q)
{
  return(mpq_cmp((mpq_ptr)&rep,(mpq_ptr)&q.rep));
}

inline int Rational::cmp_ui(const unsigned long int n, const unsigned long int d)
{
  return(mpq_cmp_ui((mpq_ptr)&rep,n,d));
}

inline int Rational::sng()
{
  return(mpq_sgn((mpq_ptr)&rep));
}

inline int Rational::equal(const Rational q)
{
  return(mpq_equal((mpq_ptr)&rep,(mpq_ptr)&q.rep));
}

inline int Rational::zero_p()
{
  return((mpq_sgn((mpq_ptr)&rep)) == 0);
}



// -------------- Conversions 

inline double Rational::get_d()
{
  return(mpq_get_d((mpq_ptr)&rep));
}

inline void Rational::set_d(const double d)
{
  mpq_set_d((mpq_ptr)&rep,d);
}


// -------------- Rational arithmetic 

inline void Rational::neg(Rational& q)
{
  mpq_neg((mpq_ptr)&rep, (mpq_ptr)&q.rep);
}

inline void Rational::add(const Rational& p, const Rational& q)
{
  mpq_add((mpq_ptr)&rep, (mpq_ptr)&p.rep, (mpq_ptr)&q.rep);
}


inline void Rational::mul(const Rational& p, const Rational& q)
{
  mpq_mul((mpq_ptr)&rep, (mpq_ptr)&p.rep, (mpq_ptr)&q.rep);
}

inline void Rational::mul_2ui(const Rational& p, const unsigned long int k)
{
  mpq_mul_2exp((mpq_ptr)&rep, (mpq_ptr)&p.rep, k);
}


inline void Rational::sub(const Rational& p, const Rational& q)
{
  mpq_sub((mpq_ptr)&rep, (mpq_ptr)&p.rep, (mpq_ptr)&q.rep);
}


inline void Rational::div(const Rational& p, const Rational& q)
{
  mpq_div((mpq_ptr)&rep, (mpq_ptr)&p.rep, (mpq_ptr)&q.rep);
}


inline void Rational::div_2ui(const Rational& p, const unsigned long int k)
{
  mpq_div_2exp((mpq_ptr)&rep, (mpq_ptr)&p.rep, k);
}


inline void Rational::abs(Rational& q)
{
  mpq_abs((mpq_ptr)&rep, (mpq_ptr)&q.rep);
}



inline void Rational::submul(const Rational& p, const Rational& q)
{
  mpq_t tmp;
  mpq_init(tmp);
  mpq_mul(tmp, (mpq_ptr)&p.rep, (mpq_ptr)&q.rep);
  mpq_sub((mpq_ptr)&rep, (mpq_ptr)&rep,  (mpq_ptr)&tmp);
  mpq_clear(tmp);
}


inline void Rational::addmul(const Rational& p, const Rational& q)
{
  mpq_t tmp;
  mpq_init(tmp);
  mpq_mul(tmp, (mpq_ptr)&p.rep, (mpq_ptr)&q.rep);
  mpq_add((mpq_ptr)&rep, (mpq_ptr)&rep,  (mpq_ptr)&tmp);
  mpq_clear(tmp);
}


inline void Rational::fma(const Rational& a, const Rational& p, const Rational& q)
{
  mpq_t tmp;
  mpq_init(tmp);
  mpq_mul(tmp, (mpq_ptr)&a.rep, (mpq_ptr)&p.rep);
  mpq_add((mpq_ptr)&rep, (mpq_ptr)&tmp,  (mpq_ptr)&q.rep);
  mpq_clear(tmp);
}


inline void Rational::fms(const Rational& a, const Rational& p, const Rational& q)
{
  mpq_t tmp;
  mpq_init(tmp);
  mpq_mul(tmp, (mpq_ptr)&a.rep, (mpq_ptr)&p.rep);
  mpq_sub((mpq_ptr)&rep, (mpq_ptr)&tmp,  (mpq_ptr)&q.rep);
  mpq_clear(tmp);
}


// -------------- Integer functions 

inline void Rational::num(mpz_t n)
{
  mpq_get_num(n,(mpq_ptr)&rep);
}

inline void Rational::num(Z_NR<mpz_t>& n)
{
  mpz_t nn;
  mpz_init(nn);
  mpq_get_num(nn,(mpq_ptr)&rep);
  n.set(nn);
  mpz_clear(nn);
}

inline void Rational::den(mpz_t n)
{
  mpq_get_den(n,(mpq_ptr)&rep);
}


inline void Rational::den(Z_NR<mpz_t>& n)
{
  mpz_t nn;
  mpz_init(nn);
  mpq_get_den(nn,(mpq_ptr)&rep);
  n.set(nn);
  mpz_clear(nn);
}


inline void Rational::set_num(const mpz_t n)
{
  mpq_set_num((mpq_ptr)&rep,n);
}

inline void Rational::set_num(Z_NR<mpz_t> n)
{
  mpq_set_num((mpq_ptr)&rep,n.GetData());
}


inline void Rational::set_den(const mpz_t n)
{
  mpq_set_den((mpq_ptr)&rep,n);
}

inline void Rational::set_den(Z_NR<mpz_t> n)
{
  mpq_set_den((mpq_ptr)&rep,n.GetData());
}


inline void Rational::rnd(const Rational& p)
{
  mpz_t q,r;
  mpq_t x;

  mpz_init(q);
  mpz_init(r);
  mpq_init(x);

  int s;
  s=mpq_sgn((mpq_ptr)&p.rep);
  mpq_abs(x,(mpq_ptr)&p.rep);

  mpz_tdiv_qr (q, r, mpq_numref(x), mpq_denref(x));
  mpz_mul_2exp(r,r,1);

  if (mpz_cmp(r, mpq_denref(x)) > 0)  mpz_add_ui(q,q,1);

  if (s < 0) mpz_neg(q,q); 
  
  mpq_set_si((mpq_ptr)&rep,1,1);
  mpq_set_num((mpq_ptr)&rep,q);
  

  mpz_clear(q);
  mpz_clear(r);
  mpq_clear(x);
}



// -------------- Input/Output
 


inline long int Rational::read(FILE *i)
{
  long int s; 
  s=mpq_inp_str((mpq_ptr)&rep,i,10);
  mpq_canonicalize((mpq_ptr)&rep);
  return(s);
}


inline long int Rational::read()
{
  long int s;
  s=mpq_inp_str((mpq_ptr)&rep,stdin,10);  
  mpq_canonicalize((mpq_ptr)&rep);
  return(s);
}


inline long int Rational::print(FILE *o)
{
  return(mpq_out_str(o,10,(mpq_ptr)&rep));
}

inline long int Rational::print()
{
  return(mpq_out_str(stdout,10,(mpq_ptr)&rep));
}

inline void Rational::printerr()
{
  mpq_out_str(stderr,10,(mpq_ptr)&rep);
}



#endif
