# A class for session management. CKSession objects have a hash of
# arbitary objects and information about browser name, IP address, etc.
# However, you can't set objects that can't be marshal ( IO, Proc, etc. )
# to the session with default database manager CKSessionStore::FileStore.
#
# == Programming Topics
# * SessionManagement[www.spice-of-life.net/download/cgikit/en/userguide/session.html]
class CKSession
	class << self
		def session_id?( session_id )
			if ( session_id == nil ) or ( session_id == "" ) then
				false
			elsif session_id =~ /([0-9A-Za-z]*)/ then
				true
			else
				false
			end
		end

		def create_session_id
			require 'digest/md5'
			md5 = Digest::MD5::new
			md5.update Time.now.to_s
			md5.update rand(0).to_s
			md5.update $$.to_s
			md5.hexdigest[0,16]
		end
	end

	# A hash of arbitary objects.
	attr_accessor :session

	# Session ID.
	attr_accessor :session_id

	# Creation time of the session.
	attr_accessor :creation_time

	# Last accessed time for the session.
	attr_accessor :last_accessed_time

	# Name of browser.
	attr_accessor :user_agent

	# IP address.
	attr_accessor :remote_addr


	def initialize( session_id = nil )
		unless CKSession.session_id? session_id then
			session_id     = CKSession.create_session_id
		end

		@session            = {}
		@session_id         = session_id
		@clear              = false
		@non_access_time    = 0
		@creation_time      = Time.new
		@last_accessed_time = Time.new
	end

	public

	def fetch( key )
		restore unless @session
		@session[key]
	end

	def store( key, value )
		restore unless @session
		@session[key] = value
	end

	alias []  fetch
	alias []= store

	def remove( key )
		@session.delete key
	end

	def clear
		@session.clear
		@clear = true
	end

	# Returns true if "clear" method of the session is called.
	def clear?
		@clear
	end

	# Returns true if the browser is equal to one when the session created.
	def user_agent?( user_agent )
		if @user_agent == user_agent then
			true
		else
			false
		end
	end

	# Returns true if the IP address is equal to one when the session created.
	def remote_addr?( remote_addr )
		if @remote_addr == remote_addr then
			true
		else
			false
		end
	end

	# Returns true if the session isn't expired.
	def timeout?( timeout )
		non_access_time = @last_accessed_time - @creation_time

		if timeout == 0 then
			false
		elsif timeout <= non_access_time then
			true
		else
			false
		end
	end
end


# CKSessionStore is a class for saving session with database manager.
# However this class is for internal use. Use methods of CKApplication to
# manage sessions.
#
# == Customizing database manager
# Database manager objects, such as CKSessionStore::FileStore, save sessions.
# The objects has these 3 methods, implement the methods if you develop or
# customize database manager class.
#
# <b>save</b>::    Saves the session.
# <b>clear</b>::   Clear the session.
# <b>restore</b>:: Returns session restored from the saved. 
#
class CKSessionStore
	attr_accessor :database_manager

	def initialize( application )
		@application      = application
		@database_manager = @application.database_manager.new @application
	end

	def save( session )
		if session.clear? then
			@database_manager.clear session.session_id
		else
			@database_manager.save session
		end
	end

	# Restore a session.
	# Returns a restored session or nil if the session ID is not existed.
	def restore( session_id )
		@database_manager.restore session_id
	end

	def clear( session_id )
		@database_manager.clear session_id
	end


	# A class that saves sessions to files by marshaling.
	class FileStore
		TMPDIR = 'session'

		def initialize( application )
			@application = application
			@tmpdir      = File.join(@application.tmpdir, TMPDIR).untaint
		end

		def save( session )
			unless FileTest.directory? @tmpdir
				require 'ftools'
				File.makedirs @tmpdir
			end

			CKFileLock.exclusive_lock( _tmpfile(session.session_id) ) do | file |
				Marshal.dump( session, file )
			end
		end

		# Restore a session.
		#
		# <b>Returns</b>
		#
		# A restored session or nil if the session ID is not existed.
		def restore( session_id )
			session = nil
			if _exist? session_id then
				CKFileLock.shared_lock( _tmpfile(session_id) ) do | file |
					session = Marshal.load file
				end
				session.last_accessed_time = Time.new
			end
			session
		end

		private
		def _tmpfile( session_id )
			File.join( @tmpdir, session_id ).untaint
		end

		def _exist?( session_id )
			if FileTest.exist? _tmpfile( session_id )
				true
			else
				false
			end
		end

		public
		def clear( session_id )
			if FileTest.exist? _tmpfile( session_id )
				File.delete _tmpfile( session_id )
			end
		end
	end
end


