# -*- coding: utf-8 -*-
#
# Authors: Natalia Bidart <nataliabidart@gmail.com>
#          Facundo Batista <facundo@taniquetil.com.ar>
#
# Copyright 2010-2011 Chicharreros
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for magicicada main UI."""

import itertools
import logging
import os
import time

from functools import wraps

import gtk
import pango

from magicicada import syncdaemon
from magicicada.gui.gtk.main import (
    MagicicadaUI,
    MetadataDialog,
    NOT_SYNCHED_PATH,
)
from magicicada.gui.gtk.tests import BaseTestCase, FakedSyncdaemon


# It's ok to access private data in the test suite
# pylint: disable=W0212

# Arguments number differs from overridden method
# pylint: disable=W0221

# Instance of 'A' has no 'y' member
# pylint: disable=E1101

# Instance of 'A' has no 'y' member (but some types could not be inferred)
# pylint: disable=E1103


class MagicicadaUITestCase(BaseTestCase):
    """UI test cases for basic state."""

    def test_init_creates_sd_instance(self):
        """SyncDaemon instance is created at creation time."""
        self.assertTrue(self.ui.sd is not None)
        self.assertTrue(isinstance(self.ui.sd, FakedSyncdaemon))

    def test_destroy_shutdowns_sd_instance(self):
        """SyncDaemon instance is shutdown at destroy time."""
        self.patch(self.ui.sd, 'shutdown', self._set_called)
        self.ui.on_main_window_destroy(self.ui.main_window)
        self.assertTrue(self._called,
                        'syncdaemon.shutdown must be called at destroy time.')

    def test_main_window_is_visible(self):
        """UI can be created and main_window is visible."""
        self.assertTrue(self.ui.main_window.get_visible())

    def test_main_window_have_correct_icon_list(self):
        """Every window has the icon set."""
        self.assertEqual(len(self.ui.main_window.get_icon_list()),
                         len(self.ui._icons.values()))

    def test_every_window_has_correct_list(self):
        """The default icon list is set."""
        self.patch(gtk, 'window_set_default_icon_list', self._set_called)
        MagicicadaUI()
        self.assertEqual(len(self._called[0]), len(self.ui._icons.values()))

    def test_start_connect_are_visible(self):
        """Start and Connect buttons are visible."""
        self.assertTrue(self.ui.start.get_visible())
        self.assertTrue(self.ui.start.is_sensitive())

        self.assertTrue(self.ui.connect.get_visible())
        self.assertFalse(self.ui.connect.is_sensitive())

    def test_stop_disconnect_are_not_visible(self):
        """Start and Connect buttons are visible."""
        self.assertFalse(self.ui.stop.get_visible())
        self.assertFalse(self.ui.disconnect.get_visible())

    def test_indicators_are_non_sensitive(self):
        """Test default sensitivity for indicators."""
        self.assertFalse(self.ui.is_started.is_sensitive())
        self.assertFalse(self.ui.is_connected.is_sensitive())
        self.assertFalse(self.ui.is_online.is_sensitive())

    def test_update_is_called_at_startup(self):
        """Update is called at startup."""
        self.patch(MagicicadaUI, 'update', self._set_called)
        self.ui = MagicicadaUI()
        self.assertTrue(self._called,
                        'update was called at startup.')

    def test_update_statusicon_idle(self):
        """Status icon is updated to idle."""
        self.ui.update(state=syncdaemon.STATE_IDLE)
        pixbuf_set = self.ui.status_icon.get_pixbuf()
        self.assertEqual(pixbuf_set, self.ui._status_icons['idle'])

    def test_update_statusicon_working(self):
        """Status icon is updated to working."""
        self.ui.update(state=syncdaemon.STATE_WORKING)
        pixbuf_set = self.ui.status_icon.get_pixbuf()
        self.assertEqual(pixbuf_set, self.ui._status_icons['working'])

    def test_update_statusicon_alert(self):
        """Status icon is updated to alert."""
        self.ui.update(state=syncdaemon.STATE_ALERT)
        pixbuf_set = self.ui.status_icon.get_pixbuf()
        self.assertEqual(pixbuf_set, self.ui._status_icons['alert'])


class ClickedTestCase(BaseTestCase):
    """UI test cases."""

    def test_on_start_clicked(self):
        """Test on_start_clicked."""
        self.ui.on_start_clicked(self.ui.start)

        self.assertTrue(self.ui.start.get_visible())
        self.assertFalse(self.ui.start.is_sensitive())
        self.assertFalse(self.ui.stop.get_visible())

        self.assert_indicator_loading(self.ui.is_started)
        self.assert_indicator_disabled(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_on_start_clicked_starts_syncdaemon(self):
        """Test on_start_clicked."""
        self.patch(self.ui.sd, 'start', self._set_called)
        self.ui.on_start_clicked(self.ui.start)
        self.assertTrue(self._called, 'syncdaemon.start was called.')

    def test_on_connect_clicked(self):
        """Test on_connect_clicked."""
        self.do_start()  # need to be started
        self.ui.on_connect_clicked(self.ui.connect)

        self.assertTrue(self.ui.connect.get_visible())
        self.assertFalse(self.ui.connect.is_sensitive())
        self.assertFalse(self.ui.disconnect.get_visible())

        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_loading(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_on_connect_clicked_connects_syncdaemon(self):
        """Test on_connect_clicked."""
        self.patch(self.ui.sd, 'connect', self._set_called)
        self.ui.on_connect_clicked(self.ui.connect)
        self.assertTrue(self._called, 'syncdaemon.connect was called.')

    def test_on_stop_clicked(self):
        """Test on_stop_clicked."""
        self.do_start()
        assert not self.ui.widget_enabled(self.ui.disconnect)
        self.patch(self.ui, 'on_disconnect_clicked', self._set_called)
        self.ui.on_stop_clicked(self.ui.stop)

        self.assertFalse(self._called, 'on_disconnect_clicked was not called.')

        self.assertFalse(self.ui.start.get_visible())
        self.assertTrue(self.ui.stop.get_visible())
        self.assertFalse(self.ui.stop.is_sensitive())

        self.assertTrue(self.ui.connect.get_visible())
        self.assertFalse(self.ui.connect.is_sensitive())
        self.assertFalse(self.ui.disconnect.get_visible())

    def test_on_stop_clicked_if_connected(self):
        """Test on_stop_clicked."""
        self.do_connect()
        self.patch(self.ui, 'on_disconnect_clicked', self._set_called)
        self.ui.on_stop_clicked(self.ui.stop)

        self.assertTrue(self._called, 'on_disconnect_clicked was called.')

    def test_on_stop_clicked_stops_syncdaemon(self):
        """Test on_stop_clicked."""
        self.patch(self.ui.sd, 'quit', self._set_called)
        self.ui.on_stop_clicked(self.ui.stop)
        self.assertTrue(self._called, 'syncdaemon.quit was called.')

    def test_on_disconnect_clicked(self):
        """Test on_disconnect_clicked."""
        self.do_connect()
        self.ui.on_disconnect_clicked(self.ui.disconnect)

        self.assertFalse(self.ui.connect.get_visible())
        self.assertTrue(self.ui.disconnect.get_visible())
        self.assertFalse(self.ui.disconnect.is_sensitive())

    def test_on_disconnect_clicked_disconnects_syncdaemon(self):
        """Test on_disconnect_clicked."""
        self.patch(self.ui.sd, 'disconnect', self._set_called)
        self.ui.on_disconnect_clicked(self.ui.disconnect)
        self.assertTrue(self._called, 'syncdaemon.disconnect was called.')


class MagicicadaUISystrayIconTestCase(BaseTestCase):
    """UI test cases for the systray icon."""

    def test_main_window_is_hid_when_icon_clicked(self):
        """Main window is hid when the systray icon is clicked."""
        self.ui.on_status_icon_activate(self.ui.status_icon)
        self.assertFalse(self.ui.main_window.get_visible(),
                         'main_window should be invisible when icon clicked.')

    def test_main_window_is_shown_when_clicked_after_hidden(self):
        """Main window is shown when the icon is clicked after hidden."""
        self.ui.on_status_icon_activate(self.ui.status_icon)  # hide
        self.ui.on_status_icon_activate(self.ui.status_icon)  # show
        msg = 'main_window should be visible when icon clicked after hidden.'
        self.assertTrue(self.ui.main_window.get_visible(), msg)


class StatusTestCase(BaseTestCase):
    """UI test cases for the status label."""

    def setUp(self):
        """Init."""
        super(StatusTestCase, self).setUp()
        name = 'TEST'
        description = 'the status for testing'
        connection = 'Funny funny connection'
        queues = 'The queues are hilarious'
        self.kwargs = dict(name=name, description=description,
                           connection=connection, queues=queues)

    def assert_status_label_correct(self, name=None, description=None,
                                    connection=None, queues=None,
                                    expected=None):
        """Test that the status label is of the form name: description."""
        if expected is None:
            assert name is not None
            assert description is not None
            assert connection is not None
            assert queues is not None
            values = (name, description, queues, connection)
            expected = self.ui.STATUS_JOINER.join(values)

        actual = self.ui.status_label.get_text()
        msg = 'status label test must be "%s" (got "%s" instead).'
        self.assertEqual(expected, actual, msg % (expected, actual))

    def test_callback_is_connected(self):
        """Status callback is connected."""
        self.assertEqual(self.ui.sd.status_changed_callback,
                         self.ui.on_status_changed,
                         'status_changed callback must be set.')

    def test_status_label_ellipsizes(self):
        """The status label ellipsizes."""
        expected = pango.ELLIPSIZE_END
        actual = self.ui.status_label.get_ellipsize()
        self.assertEqual(expected, actual,
                         'label ellipsizes is ELLIPSIZE_END.')

    def test_on_status_changed_updates_status_label(self):
        """On status changed, the status label is updated."""
        self.ui.sd.current_state.set(is_started=True, **self.kwargs)
        # usual case, all values are defined
        self.ui.on_status_changed(**self.kwargs)
        self.assert_status_label_correct(**self.kwargs)

    def test_on_status_changed_updates_status_label_even_on_weird_cases(self):
        """On status changed, the status label is updated."""
        keywords = ('name', 'description', 'queues', 'connection')  # order
        for attr in keywords:
            old_value = self.kwargs[attr]

            # some weird cases: attr is '' or None
            for value in ('', None):
                self.kwargs[attr] = value
                self.ui.sd.current_state.set(is_started=True, **self.kwargs)
                self.ui.on_status_changed(**self.kwargs)
                others = (self.kwargs[k] for k in keywords if k != attr)
                expected = self.ui.STATUS_JOINER.join(others)
                self.assert_status_label_correct(expected=expected)

            self.kwargs[attr] = old_value

    def test_on_status_changed_updates_toolbar(self):
        """On status changed, the whole toolbar is updated."""
        self.patch(self.ui, 'update', self._set_called)
        self.ui.on_status_changed(**self.kwargs)
        self.assertEqual(self._called, ((), self.kwargs))

    def test_update_is_correct_for_status_label(self):
        """Correctly updates the status label."""
        self.ui.sd.current_state.set(**self.kwargs)
        self.ui.update()
        self.assert_status_label_correct(**self.kwargs)

    def test_on_stopped_updates_status_label(self):
        """On SD stoppped, the UI is updated."""
        self.ui.sd.current_state.set(**self.kwargs)
        self.ui.on_stopped()
        self.assert_status_label_correct(**self.kwargs)

    def test_status_label_default_if_not_started(self):
        """Status label is the default if not started."""
        self.assert_status_label_correct(expected=self.ui.STATUS['initial'])

    def test_status_label_empty_if_started_and_no_name_nor_desc(self):
        """Status label is empty if started but no name and no description."""
        self.do_start()
        self.ui.on_status_changed(name=None, description=None)
        self.assert_status_label_correct(expected='')

    def test_status_label_is_updated_on_started_and_on_stopped(self):
        """Status label is updated on_started."""
        self.ui.sd.current_state.set(**self.kwargs)
        self.do_start()
        self.assert_status_label_correct(**self.kwargs)

        self.kwargs['name'] = 'CHANGED'
        self.ui.sd.current_state.set(**self.kwargs)

        self.ui.on_stop_clicked(self.ui.stop)
        self.ui.on_stopped()
        self.assert_status_label_correct(**self.kwargs)


class ConnectionTestCase(BaseTestCase):
    """UI test cases for connection buttons/indicators."""

    def assert_buttons_are_updated_correctly(self):
        """Test that buttons are correctly updated as per SD current state."""
        cs = self.ui.sd.current_state
        self.ui.update()

        self.assertEqual(self.ui.widget_enabled(self.ui.start),
                         not cs.is_started,
                         'start must be enabled if not started.')
        self.assertEqual(self.ui.widget_enabled(self.ui.stop),
                         cs.is_started,
                         'stop must be enabled if started.')

        if cs.is_started:
            self.assertEqual(self.ui.widget_enabled(self.ui.connect),
                             not cs.is_connected,
                             'connect must be enabled if not connected.')
            self.assertEqual(self.ui.widget_enabled(self.ui.disconnect),
                             cs.is_connected,
                             'disconnect must be enabled if connected.')
        else:
            self.assertFalse(self.ui.connect.is_sensitive(),
                             'connect must be disabled when %s' % cs)
            self.assertTrue(self.ui.connect.get_visible(),
                            'connect must be visible when %s' % cs)

    def assert_indicators_are_updated_correctly(self):
        """Test that indicators are correctly updated as per SD state."""
        cs = self.ui.sd.current_state
        self.ui.update()
        if cs.is_started and not cs.is_connected and not cs.is_online:
            self.assert_indicator_ready(self.ui.is_started)
            self.assert_indicator_disabled(self.ui.is_connected)
            self.assert_indicator_disabled(self.ui.is_online)
        elif cs.is_started and cs.is_connected and not cs.is_online:
            self.assert_indicator_ready(self.ui.is_started)
            self.assert_indicator_ready(self.ui.is_connected)
            self.assert_indicator_loading(self.ui.is_online)
        elif cs.is_started and cs.is_connected and cs.is_online:
            self.assert_indicator_ready(self.ui.is_started)
            self.assert_indicator_ready(self.ui.is_connected)
            self.assert_indicator_ready(self.ui.is_online)
        elif not cs.is_started:
            self.assert_indicator_disabled(self.ui.is_started)
            self.assert_indicator_disabled(self.ui.is_connected)
            self.assert_indicator_disabled(self.ui.is_online)

    def test_all_disabled_at_startup(self):
        """Indicators are all disabled at startup."""
        self.assert_indicator_disabled(self.ui.is_started)
        self.assert_indicator_disabled(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_callbacks_are_connected(self):
        """Connection callbacks are connected."""
        for callback in ('on_started', 'on_stopped',
                         'on_connected', 'on_disconnected',
                         'on_online', 'on_offline'):
            sd_cb = getattr(self.ui.sd, '%s_callback' % callback)
            ui_cb = getattr(self.ui, callback)
            self.assertEqual(sd_cb, ui_cb,
                             '%s callback must be set' % callback)

    def test_on_started_is_correct(self):
        """On SD started, the UI enables connect and indicator."""
        self.do_start()

        self.assertTrue(self.ui.widget_enabled(self.ui.stop))
        self.assertTrue(self.ui.widget_enabled(self.ui.connect))
        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_disabled(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_on_connected_is_correct(self):
        """On SD connected, the UI enables indicator."""
        self.do_connect()

        self.assertTrue(self.ui.widget_enabled(self.ui.disconnect))
        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_ready(self.ui.is_connected)
        self.assert_indicator_loading(self.ui.is_online)

    def test_on_online_is_correct(self):
        """On SD online, the UI enables indicator."""
        self.do_connect()

        self.ui.on_online()

        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_ready(self.ui.is_connected)
        self.assert_indicator_ready(self.ui.is_online)

    def test_on_stopped_is_correct(self):
        """On SD stopped, the UI disables stop and indicator."""
        self.do_start()
        self.ui.on_stop_clicked(self.ui.stop)

        self.ui.on_stopped()

        self.assertTrue(self.ui.start.is_sensitive())
        self.assertTrue(self.ui.start.get_visible())
        self.assert_indicator_disabled(self.ui.is_started)
        self.assert_indicator_disabled(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_on_disconnected_is_correct(self):
        """On SD disconnected, the UI disables connect and indicator."""
        self.do_connect()
        self.ui.on_disconnect_clicked(self.ui.disconnect)

        self.ui.on_disconnected()

        self.assertTrue(self.ui.connect.is_sensitive())
        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_disabled(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_on_offline_is_correct(self):
        """On SD offline, the UI disables indicator."""
        self.do_connect()

        self.ui.on_offline()

        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_ready(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_update_is_correct_for_buttons(self):
        """Correctly updates the buttons state."""
        for s, c, o in itertools.product((True, False), repeat=3):
            self.ui.sd.current_state.set(is_started=s, is_connected=c,
                                         is_online=o)
            self.assert_buttons_are_updated_correctly()

    def test_update_is_correct_for_indicators(self):
        """Correctly updates the indicators state."""
        for s, c, o in itertools.product((True, False), repeat=3):
            self.ui.sd.current_state.set(is_started=s, is_connected=c,
                                         is_online=o)
            self.assert_indicators_are_updated_correctly()


class ToolbarTestCase(BaseTestCase):
    """Toolbar test case."""

    def assert_widget_availability(self, enabled=True,
                                   public_files_enabled=True):
        """Check button availability according to 'enabled'."""
        widget = self.ui.toolbar
        self.assertTrue(widget.get_visible(), 'Should be visible.')
        # all children should be visible

        for n_button in xrange(self.ui.toolbar.get_n_items()):
            button = self.ui.toolbar.get_nth_item(n_button)
            self.assertTrue(button.get_visible(),
                            'Children should be visible.')

        sensitive = widget.is_sensitive()
        msg = 'Should %sbe sensitive.' % ('' if enabled else 'not ')
        self.assertTrue(sensitive if enabled else not sensitive, msg)

        for n_button in xrange(self.ui.toolbar.get_n_items()):
            button = self.ui.toolbar.get_nth_item(n_button)
            sensitive = button.is_sensitive()

            if button is self.ui.public_files:
                expected = public_files_enabled
            else:
                expected = enabled

            msg = 'Button %i should %sbe sensitive.'
            self.assertTrue(sensitive if expected else not sensitive,
                            msg % (n_button, '' if expected else 'not '))

    def test_initial_data_ready_callback_connected(self):
        """The callback 'on_initial_data_ready' is connected to SD."""
        self.assertEqual(self.ui.sd.on_initial_data_ready_callback,
                         self.ui.on_initial_data_ready,
                         "on_initial_data_ready should be connected.")

    def test_initial_online_data_ready_callback_connected(self):
        """The callback 'on_initial_online_data_ready' is connected to SD."""
        self.assertEqual(self.ui.sd.on_initial_online_data_ready_callback,
                         self.ui.on_initial_online_data_ready,
                         "on_initial_online_data_ready should be connected.")

    def test_disabled_until_initial_data_ready(self):
        """Folders and shares are disabled until data ready."""
        # disabled at startup
        self.assert_widget_availability(enabled=False,
                                        public_files_enabled=False)

        # enabled when initial data ready
        self.ui.on_initial_data_ready()
        self.assert_widget_availability(public_files_enabled=False)

    def test_public_files_disabled_until_initial_online_data_ready(self):
        """Widget is disabled until initial online data ready."""
        # disabled at startup
        self.assert_widget_availability(enabled=False,
                                        public_files_enabled=False)

        # all enabled but public_files when initial data ready
        self.ui.on_initial_data_ready()
        self.assert_widget_availability(public_files_enabled=False)

        # all enabled when initial online data ready
        self.ui.on_initial_online_data_ready()
        self.assert_widget_availability()

    def test_enabled_until_stopped(self):
        """Folders and shares are enabled until offline."""
        self.ui.on_initial_data_ready()

        self.do_connect()
        self.assert_widget_availability(public_files_enabled=False)

        self.ui.on_online()
        self.assert_widget_availability(public_files_enabled=False)

        self.ui.on_initial_online_data_ready()
        self.assert_widget_availability()

        self.ui.on_offline()
        self.assert_widget_availability()

        self.ui.on_disconnect_clicked(self.ui.disconnect)
        self.ui.on_disconnected()
        self.assert_widget_availability()

        # disabled when stopped
        self.ui.on_stop_clicked(self.ui.stop)
        self.assert_widget_availability(enabled=False,
                                        public_files_enabled=False)
        self.ui.on_stopped()
        self.assert_widget_availability(enabled=False,
                                        public_files_enabled=False)

    def test_on_stopped_updates_availability(self):
        """On SD stoppped, the UI disables the toolbar."""
        self.ui.on_initial_online_data_ready()
        self.ui.on_stopped()
        self.assert_widget_availability(enabled=False,
                                        public_files_enabled=False)


class MetadataTestCase(BaseTestCase):
    """UI test cases for metadata display."""

    TEST_FILE = os.path.join('/tmp', 'metadata-test.txt')
    TEST_DIR = os.path.join('/tmp', 'metadata-test-dir')

    def setUp(self):
        """Init."""
        super(MetadataTestCase, self).setUp()

        # no need for the file_chooser to actually run, it works.
        self.patch(self.ui.file_chooser, 'run', lambda: gtk.RESPONSE_CLOSE)

        # create temp file and dir
        open(self.TEST_FILE, 'w').close()
        self.addCleanup(os.remove, self.TEST_FILE)
        os.mkdir(self.TEST_DIR)
        self.addCleanup(os.rmdir, self.TEST_DIR)

        self._file_chooser_path = None
        self.metadata = {
            'raw_result': dict(bla='ble', foo='bar'),
            'stat': dict(st_size=123, st_mtime=123),
            'path': 'path',
            'changed': 'changed',
        }

        # no need for the file_chooser to actually run, it works.
        self.patch(self.ui.file_chooser, 'run',
                   lambda: gtk.FILE_CHOOSER_ACTION_OPEN)

        # store whatever file was set
        self.patch(self.ui.file_chooser, 'set_filename',
                   lambda path: setattr(self, '_file_chooser_path', path))

        # return the stored path
        self.patch(self.ui.file_chooser, 'get_filename',
                   lambda: getattr(self, '_file_chooser_path'))

    def assert_visibility(self, md_dialog, dialog, info, spinner):
        """Check the visibility for dialog, text_view and spinner."""
        msg = '%s visibility should be %s (got %s instead).'
        visible = md_dialog.dialog.get_visible()
        self.assertEqual(dialog, visible,
                         msg % ('dialog', dialog, visible))

        visible = md_dialog.filepath_hbox.get_visible()
        self.assertEqual(info, visible,
                         msg % ('filepath_hbox', info, visible))

        visible = md_dialog.state_hbox.get_visible()
        self.assertEqual(info, visible,
                         msg % ('state_hbox', info, visible))

        visible = md_dialog.details_expander.get_visible()
        self.assertEqual(info, visible,
                         msg % ('details_expander', info, visible))

        visible = md_dialog.spinner.get_visible()
        self.assertEqual(spinner, visible,
                         msg % ('spinner', spinner, visible))

    def test_metadata_close_hides_the_dialog(self):
        """Test metadata close button emits RESPONSE_CLOSE when clicked."""
        md = MetadataDialog()
        self.patch(md.dialog, 'hide', self._set_called)

        md.close_button.clicked()
        self.assertEqual(self._called, ((), {}),
                         "MetadataDialog.hide should be called.")

    def test_file_chooser_open_emits_response_ok(self):
        """Test volume close button emits RESPONSE_CLOSE when clicked."""
        self.patch(self.ui.file_chooser, 'response', self._set_called)

        self.ui.file_chooser_open.clicked()

        msg = 'file_chooser_open should emit %s (got %s instead).'
        expected = gtk.FILE_CHOOSER_ACTION_OPEN
        self.assertEqual(self._called, ((expected,), {}),
                         msg % (expected, self._called))

    def test_on_metadata_clicked(self):
        """Test on_metadata_clicked."""
        self.ui._u1_root = os.path.dirname(self.TEST_FILE)
        self.ui.file_chooser.set_filename(self.TEST_FILE)

        self.ui.on_metadata_clicked(self.ui.metadata)

        # file_chooser must be visible on metadata clicked.
        self.assertEqual(self.TEST_FILE, self.ui.file_chooser.get_filename(),
                         'filename returned by file chooser must be correct.')

        # dialog must exist now
        dialog = self.ui.metadata_dialogs[self.TEST_FILE]

        # metadata_dialog is enabled and shows the loading animation
        self.assert_visibility(dialog, dialog=True, info=False, spinner=True)
        self.assertIsInstance(dialog.spinner, gtk.Spinner)
        self.assertTrue(dialog.spinner.get_property('active'),
                        'metadata_spinner is active.')

        # check that the metadata was asked to the SD
        self.assertEqual(1, len(self.ui.sd._meta_paths))
        self.assertEqual(self.ui.sd._meta_paths[0], self.TEST_FILE)
        # SD will eventually callback us with the metadata
        self.ui.on_metadata_ready(self.TEST_FILE, self.metadata)

        # metadata_dialog is enabled and shows the metadata
        self.assert_visibility(dialog, dialog=True, info=True, spinner=False)

        # user closes the dialog
        dialog.close_button.clicked()

        # dialog was closed already
        visible = dialog.dialog.get_visible()
        self.assertFalse(visible, 'metadata_dialog should not be visible.')

    def test_metadata_dialog_properties(self):
        """The metadata dialog has correct properties."""
        # title and modal
        title = 'Metadata for %s' % self.TEST_FILE
        md = MetadataDialog()
        md.run()
        md.got_metadata(self.TEST_FILE, self.metadata)
        self.assert_dialog_properties(dialog=md.dialog,
                                      position=gtk.WIN_POS_MOUSE,
                                      title=title, modal=False)

        # text should be wrapped
        actual = md.detailed_info_textview.get_wrap_mode()
        msg = 'wrap mode for view must be gtk.WRAP_WORD (got %s instead).'
        self.assertEqual(gtk.WRAP_WORD, actual, msg % actual)

        # labels should be selectable
        self.assertTrue(md.path_label.get_selectable())
        self.assertTrue(md.basic_info_label.get_selectable())

    def test_callback_is_connected(self):
        """Metadata ready callback is connected."""
        self.assertEqual(self.ui.sd.on_metadata_ready_callback,
                         self.ui.on_metadata_ready,
                         'on_metadata_ready_callback callback must be set.')

    def test_file_chooser_is_hidden_at_startup(self):
        """File chooser exists but is not visible."""
        self.assertFalse(self.ui.file_chooser.get_visible(),
                         'file_chooser must be hidden by default.')

    def test_filename_is_used_only_if_open_clicked(self):
        """Filename is used only if user clicked open."""
        # no need for the file_chooser to actually run, it works.
        self.patch(self.ui.file_chooser, 'run', lambda: gtk.RESPONSE_CLOSE)
        self.patch(self.ui.sd, 'get_metadata', self._set_called)

        self.ui.on_metadata_clicked(self.ui.metadata)

        self.assertFalse(self._called,
                        'get_metadata should not be called if no file chosen.')

    def test_filename_is_stored_if_open_was_clicked(self):
        """Filename is stored in the metadata dicts if user clicked open."""
        self.assertEqual(self.ui.metadata_dialogs, {},
                        'no dialogs in the ui.')

        self.ui._u1_root = os.path.dirname(self.TEST_FILE)
        self.ui.file_chooser.set_filename(self.TEST_FILE)
        self.ui.on_metadata_clicked(self.ui.metadata)

        self.assertEqual([self.TEST_FILE], self.ui.metadata_dialogs.keys(),
                         'metadata dict keys should be what the user choose.')

    def test_on_metadata_ready(self):
        """Callback on_metadata_ready executes the dialog method."""
        path = 'bla'
        md = MetadataDialog()
        called = []
        md.got_metadata = lambda *a: called.extend(a)
        self.ui.metadata_dialogs[path] = md
        self.ui.on_metadata_ready(path, self.metadata)
        self.assertEqual(called, [path, self.metadata])

    def test_on_metadata_ready_doesnt_update_if_last_path_doesnt_match(self):
        """Callback on_metadata_ready updates the metadata_view."""
        path = 'bla'
        assert path not in self.ui.metadata_dialogs
        self.ui.on_metadata_ready(path, self.metadata)
        self.assertTrue(self.memento.check_info("on_metadata_ready", path,
                                                "not in stored paths"))

    def test_two_metadata_windows(self):
        """More than one metadata window is allowed."""
        self.patch(self.ui.file_chooser, 'run',
                   lambda: gtk.FILE_CHOOSER_ACTION_OPEN)

        path1 = os.path.abspath(self.mktemp())
        open(path1, 'w').close()
        assert os.path.exists(path1)
        meta1 = dict(raw_result={'value': 'Lorem ipsum dolor sit amet.'})

        path2 = os.path.abspath(self.mktemp())
        open(path2, 'w').close()
        assert os.path.exists(path2)
        meta2 = dict(raw_result={'value': 'Etiam iaculis congue nisl.'})

        self.ui.file_chooser.get_filename = lambda: path1
        self.ui.on_metadata_clicked(self.ui.metadata)

        self.ui.file_chooser.get_filename = lambda: path2
        self.ui.on_metadata_clicked(self.ui.metadata)

        # Check that the UI has 2 metadata dialogs
        self.assertTrue(2, len(self.ui.metadata_dialogs))
        called1 = []
        md1 = self.ui.metadata_dialogs[path1]
        md1.got_metadata = lambda *a: called1.extend(a)
        called2 = []
        md2 = self.ui.metadata_dialogs[path2]
        md2.got_metadata = lambda *a: called2.extend(a)

        # Check that the metadata was asked to the SD
        self.assertEqual(2, len(self.ui.sd._meta_paths))
        self.assertEqual(self.ui.sd._meta_paths[0], path1)
        self.assertEqual(self.ui.sd._meta_paths[1], path2)

        # SD will eventually callback us with the metadata
        self.ui.on_metadata_ready(path1, meta1)
        # SD will eventually callback us with the metadata
        self.ui.on_metadata_ready(path2, meta2)

        self.assertEqual(called1, [path1, meta1])
        self.assertEqual(called2, [path2, meta2])

        # user closes the dialog
        self.ui.metadata_dialogs[path1].close_button.clicked()
        self.ui.metadata_dialogs[path2].close_button.clicked()

    def test_filetype_icon_file(self):
        """Put the file image when it's a file."""
        md = MetadataDialog()
        md.got_metadata(self.TEST_FILE, self.metadata)
        icon, size = md.filetype_image.get_stock()
        self.assertEqual(icon, gtk.STOCK_FILE)
        self.assertEqual(size, gtk.ICON_SIZE_MENU)

    def test_filetype_icon_dir(self):
        """Put the dir image when it's a dir."""
        md = MetadataDialog()
        md.got_metadata(self.TEST_DIR, self.metadata)
        icon, size = md.filetype_image.get_stock()
        self.assertEqual(icon, gtk.STOCK_DIRECTORY)
        self.assertEqual(size, gtk.ICON_SIZE_MENU)

    def test_not_synched(self):
        """Special case: path not synched."""
        md = MetadataDialog()
        md.got_metadata(self.TEST_FILE, NOT_SYNCHED_PATH)
        self.assertTrue(md.path_label.get_visible())
        self.assertEqual(md.path_label.get_text(), self.TEST_FILE)
        self.assertTrue(md.basic_info_label.get_visible())
        self.assertEqual(md.basic_info_label.get_text(), NOT_SYNCHED_PATH)

    def test_path_label(self):
        """Check it puts the correct path in the label."""
        md = MetadataDialog()
        formatted_path = "specially formatted path"
        self.metadata['path'] = formatted_path
        assert self.TEST_FILE != formatted_path
        md.got_metadata(self.TEST_FILE, self.metadata)
        self.assertEqual(md.path_label.get_text(), formatted_path)

    def test_simple_text_no_stat(self):
        """The simple text should support not having stat."""
        md = MetadataDialog()
        self.metadata['stat'] = None
        md.got_metadata(self.TEST_FILE, self.metadata)
        simple_text = md.basic_info_label.get_text()
        self.assertFalse("Size" in simple_text)
        self.assertFalse("Modified" in simple_text)

    def test_simple_text_size(self):
        """The simple text should have a size."""
        md = MetadataDialog()
        self.metadata['stat']['st_size'] = 678
        md.got_metadata(self.TEST_FILE, self.metadata)
        simple_text = md.basic_info_label.get_text()
        self.assertTrue("Size: 678" in simple_text)

    def test_simple_text_size_broken_humanize(self):
        """Support a broken humanization."""
        md = MetadataDialog()
        broken_size = "will break humanize_bytes"
        self.metadata['stat']['st_size'] = broken_size
        md.got_metadata(self.TEST_FILE, self.metadata)
        simple_text = md.basic_info_label.get_text()
        self.assertTrue(broken_size in simple_text)

    def test_simple_text_modified(self):
        """The simple text should have a tstamp."""
        md = MetadataDialog()
        mtime = time.time()
        self.metadata['stat']['st_mtime'] = mtime
        md.got_metadata(self.TEST_FILE, self.metadata)
        simple_text = md.basic_info_label.get_text()
        should = "Modified on " + time.ctime(mtime)
        self.assertTrue(should in simple_text)

    def test_detailed_text(self):
        """The raw data is also shown."""
        md = MetadataDialog()
        md.got_metadata(self.TEST_FILE, self.metadata)
        raw = self.metadata['raw_result']
        should = '\n'.join('%s: %s' % i for i in raw.iteritems())
        buf = md.detailed_info_textview.get_buffer()
        detailed = buf.get_text(buf.get_start_iter(), buf.get_end_iter())
        self.assertTrue(should in detailed)

    def test_state_none(self):
        """Text and icon for state NONE."""
        md = MetadataDialog()
        self.metadata['changed'] = syncdaemon.CHANGED_NONE
        md.got_metadata(self.TEST_FILE, self.metadata)
        simple_text = md.basic_info_label.get_text()
        self.assertTrue("Synchronized" in simple_text)
        icon, size = md.state_image.get_stock()
        self.assertEqual(icon, gtk.STOCK_APPLY)
        self.assertEqual(size, gtk.ICON_SIZE_LARGE_TOOLBAR)

    def test_state_local(self):
        """Text and icon for state LOCAL."""
        md = MetadataDialog()
        self.metadata['changed'] = syncdaemon.CHANGED_LOCAL
        md.got_metadata(self.TEST_FILE, self.metadata)
        simple_text = md.basic_info_label.get_text()
        self.assertTrue("With local changes, uploading" in simple_text)
        icon, size = md.state_image.get_stock()
        self.assertEqual(icon, gtk.STOCK_GO_UP)
        self.assertEqual(size, gtk.ICON_SIZE_LARGE_TOOLBAR)

    def test_state_server(self):
        """Text and icon for state SERVER."""
        md = MetadataDialog()
        self.metadata['changed'] = syncdaemon.CHANGED_SERVER
        md.got_metadata(self.TEST_FILE, self.metadata)
        simple_text = md.basic_info_label.get_text()
        self.assertTrue("With server changes, downloading" in simple_text)
        icon, size = md.state_image.get_stock()
        self.assertEqual(icon, gtk.STOCK_GO_DOWN)
        self.assertEqual(size, gtk.ICON_SIZE_LARGE_TOOLBAR)

    def test_state_unknown(self):
        """Text and icon for unknow state."""
        md = MetadataDialog()
        self.metadata['changed'] = 'broken'
        md.got_metadata(self.TEST_FILE, self.metadata)
        simple_text = md.basic_info_label.get_text()
        self.assertTrue("Unknown state" in simple_text)
        self.assertFalse(md.state_image.get_visible())


def override_input_output(input_args, output_args):
    """Call 'f' but receive fixed input and return fixed output."""

    def decorator(f):
        """The decorator per se."""

        @wraps(f)
        def inner(*args, **kwargs):
            """Feed 'f' with 'input_args' and return 'output_args'."""
            f(input_args)
            return output_args

        return inner

    return decorator


class MagicicadaLoggingTestCase(BaseTestCase):
    """UI test cases for logging."""

    def test_on_status_changed_logs(self):
        """Check _on_status_changed logs properly."""
        args = ('test status', 'status description', True, False, True)
        kwargs = dict(queues='bla', connection=None)
        self.assert_function_logs(logging.getLevelName(logging.DEBUG),
                                  self.ui.on_status_changed, *args, **kwargs)

    def test_on_metadata_ready_logs(self):
        """Check on_metadata_ready logs properly."""
        args = ()
        kwargs = dict(path='test', metadata=True)
        self.assert_function_logs(logging.getLevelName(logging.DEBUG),
                                  self.ui.on_metadata_ready, *args, **kwargs)

    def test_on_initial_data_ready_logs(self):
        """Check on_initial_data_ready logs properly."""
        args = ()
        kwargs = dict(path='test', metadata=True)
        self.assert_function_logs(logging.getLevelName(logging.INFO),
                                  self.ui.on_initial_data_ready,
                                  *args, **kwargs)
