#!/usr/bin/tclsh
# Part of MCU 8051 IDE ( http://mcu8051ide.sf.net )

############################################################################
#    Copyright (C) 2007-2009 by Martin Ošmera                              #
#    martin.osmera@gmail.com                                               #
#                                                                          #
#    This program is free software; you can redistribute it and#or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

# --------------------------------------------------------------------------
# DESCRIPTION
# Implements simulator configuration dialog
# --------------------------------------------------------------------------

namespace eval simulator {

	variable win			;# ID of toplevel dialog window
	variable dialog_opened	0	;# Bool: True if this dialog is already opened

	# List of default settings
	variable defaults {
		{reverse_run_steps		10}
		{ignore_read_from_wr_only	0}
		{ignore_invalid_reti		0}
		{ignore_watchdog_reset		0}
		{ignore_stack_overflow		0}
		{ignore_stack_underflow		0}
		{ignore_invalid_ins		0}
		{ignore_invalid_IDATA		0}
		{ignore_invalid_XDATA		0}
		{ignore_invalid_BIT		0}
		{ignore_invalid_CODE		0}
		{ignore_EEPROM_WR_fail		0}
		{ignore_EEPROM_WR_abort		0}
		{ignore_invalid_USB		0}
		{ignore_invalid_UMC		0}
		{ignore_invalid_TMC		0}
		{undefined_value		2}
	}

	# Option variables
	variable reverse_run_steps		;# Int: Number of steps which can be taken back
	variable ignore_watchdog_reset		;# Bool: Ignore reset invoked by watchdog overflow
	variable ignore_read_from_wr_only	;# Bool: Ignore reading from read only register
	variable ignore_stack_overflow		;# Bool: Do not show "Stack overflow" dialog
	variable ignore_stack_underflow		;# Bool: Do not show "Stack underflow" dialog
	variable ignore_invalid_reti		;# Bool: Ignore invalid return fom interrupt
	variable ignore_invalid_ins		;# Bool: Ignore invalid instructions
	variable ignore_invalid_IDATA		;# Bool: Ignore access to unimplemented IDATA memory
	variable ignore_invalid_XDATA		;# Bool: Ignore access to unimplemented XDATA memory
	variable ignore_invalid_BIT		;# Bool: Ignore access to unimplemented bit
	variable ignore_invalid_CODE		;# Bool: Ignore access to unimplemented CODE memory
	variable ignore_EEPROM_WR_fail		;# Bool: Ignore EEPROM write failure
	variable ignore_EEPROM_WR_abort		;# Bool: Ignore EEPROM write abort
	variable ignore_invalid_USB		;# Bool: Ignore UART frame discart
	variable ignore_invalid_UMC		;# Bool: Ignore invalid UART mode change
	variable ignore_invalid_TMC		;# Bool: Ignore invalid Timer/Counter mode change
	variable undefined_value		;# Int: How to handle undefined values (0 == 0; 1 == 255; 2 == random)


	## Create the dialog
	 # @return void
	proc mkDialog {} {
		variable win		;# ID of toplevel dialog window
		variable dialog_opened	;# Bool: True if this dialog is already opened

		# Destroy the dialog if it's alredy opened
		if {$dialog_opened} {
			destroy .simulator_config_dialog
		}
		set dialog_opened 1

		# Get settings from Compiler NS
		getSettings

		# Create toplevel window
		set win [toplevel .simulator_config_dialog -class {Configuration dialog} -bg {#EEEEEE}]

		# Create window header
		label $win.header_label				\
			-compound left				\
			-image ::ICONS::32::kcmmemory		\
			-text [mc "Simulator configuration"]	\
			-font [font create	\
				-size -20]

		## Create notebook
		set nb [ttk::notebook $win.nb]
		 # Tab "Warning dialogs"
		set warnings_tab [frame $nb.warnings_tab]
		$nb add $warnings_tab -text [mc "Warning dialogs"]
		  # Tab "Other"
		set other_tab [frame $nb.other_tab]
		$nb add $other_tab -text [mc "Other"]

		#
		## Tab "Warning dialogs"
		#
		set row 0
		foreach text {
				{Ignore stack overflow}
				{Ignore stack underflow}
				{-}
				{Ignore invalid instructions}
				{Ignore watchdog overflow}
				{Ignore invalid return from interrupt}
				{Ignore reading from write only register}
				{-}
				{Ignore invalid access to IDATA/SFR}
				{Ignore invalid access to EDATA}
				{Ignore invalid access to XDATA}
				{Ignore invalid access to bit}
				{Ignore invalid access to CODE}
				{-}
				{Ignore EEPROM write failure}
				{Ignore EEPROM write abort}
				{-}
				{Ignore UART frame discard}
				{Ignore illegal UART mode change}
				{Ignore illegal Timer/Counter mode change}
			} helptext {
				{Check this to disable warning on stack overflow}
				{Check this to disable warning on stack underflow}
				{-}
				{Check this to disable warning on\ninvalid instruction}
				{Do not stop simulation on device reset\ninvoked by watchog timer overflow}
				{Do not show warning dialog when program tring to return from interrupt which has not been invoked}
				{Do not display warning dialog when\nreading from write-only register}
				{-}
				{Do not display dialog "Undefined result" when simulated program\naccessing unimplemented Internal Data Memory (IDATA) or SFR area}
				{Do not display dialog "Undefined result" when simulated program\naccessing unimplemented Expanded Data Memory (EDATA)}
				{Do not display dialog "Undefined result" when simulated program\naccessing unimplemented External Data Memory (XDATA)}
				{Do not display dialog "Undefined result" when simulated program\naccessing unimplemented bit in IDATA or SFR area}
				{Do not display dialog "Undefined result" when simulated program\naccessing unimplemented Program Memory (CODE)}
				{-}
				{Check this to disable warning on\ndata eeprom write failure}
				{Check this to disable warning on\ndata eeprom write abort}
				{-}
				{Check this to disable warning on UART frame discard}
				{Check this to disable warning on illegal UART mode change}
				{Check this to disable warning on illegal Timer/Counter mode change}
			} variable {
				ignore_stack_overflow	ignore_stack_underflow
				-
				ignore_invalid_ins	ignore_watchdog_reset
				ignore_invalid_reti	ignore_read_from_wr_only
				-
				ignore_invalid_IDATA	ignore_invalid_EDATA
				ignore_invalid_XDATA	ignore_invalid_BIT
				ignore_invalid_CODE
				-
				ignore_EEPROM_WR_fail	ignore_EEPROM_WR_abort
				-
				ignore_invalid_USB	ignore_invalid_UMC
				ignore_invalid_TMC
		} {
			incr row

			# Create separator
			if {$text == {-}} {
				grid [ttk::separator $warnings_tab.sep_$row	\
					-orient horizontal			\
				] -column 0 -row $row -columnspan 2 -sticky we -pady 5 -padx 5
				continue
			}

			# Create
			grid [Label $warnings_tab.label__$row		\
				-text $text -helptext [subst $helptext]	\
			] -row $row -column 0 -sticky w -padx 5
			grid [checkbutton $warnings_tab.chbutton_$row		\
				-variable ::configDialogs::simulator::$variable	\
			] -row $row -column 1 -sticky e -padx 5
			DynamicHelp::add $warnings_tab.chbutton_$row	\
				-text [subst $helptext]
		}
		grid columnconfigure $warnings_tab 0 -minsize 250
		grid columnconfigure $warnings_tab 1 -weight 1

		#
		# Tab "Other"
		#

		# LabelFrame: "Undefined values"
		set undefined_labelframe [ttk::labelframe $other_tab.undefined_labelframe	\
			-text [mc "Undefined values"] -padding 7				\
		]
		pack [radiobutton $undefined_labelframe.random			\
			-value 2 -text [mc "Return random value"]		\
			-variable ::configDialogs::simulator::undefined_value	\
		] -anchor w
		pack [radiobutton $undefined_labelframe.zero			\
			-value 0 -text [mc "Return zero value"]			\
			-variable ::configDialogs::simulator::undefined_value	\
		] -anchor w
		pack [radiobutton $undefined_labelframe.one			\
			-value 1 -text [mc "Return highest possible value"]	\
			-variable ::configDialogs::simulator::undefined_value	\
		] -anchor w
		pack $undefined_labelframe -fill x -padx 5 -pady 10

		# LabelFrame: "Reverse run"
		set reverse_run_labelframe [ttk::labelframe $other_tab.reverse_run_labelframe	\
			-text [mc "Reverse run"] -padding 7					\
		]
		grid [Label $reverse_run_labelframe.rrun_lbl				\
			-text [mc "Stack capacity"]					\
			-helptext [mc "Number of steps which can be taken back"]	\
		] -row 4 -column 0 -sticky w -padx 5
		grid [spinbox $reverse_run_labelframe.rrun_spinbox			\
			-from 0 -to 1000 -bg white -validate all -width 4		\
			-vcmd "::configDialogs::simulator::rrun_spinbox_val %P"		\
			-textvariable ::configDialogs::simulator::reverse_run_steps	\
		] -row 4 -column 1 -sticky we -padx 5
		DynamicHelp::add $reverse_run_labelframe.rrun_spinbox	\
			-text [mc "Number of steps which can be taken back"]
		grid columnconfigure $reverse_run_labelframe 0 -minsize 250
		grid columnconfigure $reverse_run_labelframe 1 -weight 1
		pack $reverse_run_labelframe -fill x -padx 5 -pady 10

		# Raise tab "Output"
		$nb select $warnings_tab

		# Create button frame at the bottom
		set but_frame [frame $win.button_frame]
			# Button "Defaults"
		pack [ttk::button $but_frame.but_default			\
			-text [mc "Defaults"]					\
			-command {::configDialogs::simulator::DEFAULTS}		\
		] -side left
		DynamicHelp::add $but_frame.but_default -text [mc "Reset settings to defaults"]
			# Button "Ok"
		pack [ttk::button $but_frame.but_ok				\
			-text [mc "Ok"]					\
			-compound left					\
			-image ::ICONS::16::ok				\
			-command {::configDialogs::simulator::OK}	\
		] -side right
			# Button "Cancel"
		pack [ttk::button $but_frame.but_cancel			\
			-text [mc "Cancel"]				\
			-compound left					\
			-image ::ICONS::16::button_cancel		\
			-command {::configDialogs::simulator::CANCEL}	\
		] -side right

		# Pack frames and notebook
		pack $win.header_label -side top -pady 6
		pack $nb -side top -fill both -expand 1 -padx 10
		pack $but_frame -side top -fill x -expand 1 -anchor s -padx 10 -pady 5

		# Set window attributes
		wm iconphoto $win ::ICONS::16::configure
		wm transient $win .
		wm title $win [mc "Simulator configuration - %s" ${::APPNAME}]
		wm minsize $win 380 480
		raise $win
		catch {grab $win}
		wm protocol $win WM_DELETE_WINDOW {
			::configDialogs::simulator::CANCEL
		}
		tkwait window $win
	}

	## Validate contents of spinbox in section "Reverse Run"
	 # @parm String content - String to validate
	 # @return Bool - validation result (0 == failed; 1 == successfull)
	proc rrun_spinbox_val {content} {
		if {![string is digit $content]} {
			return 0
		}
		if {$content > 1000} {
			return 0
		}
		return 1
	}

	## Set configuration variable
	 # This function is unsafe -- you must be sure by the given arguments
	 # @parm String variable	- variable to set
	 # @parm Mixed value		- new value
	proc set_variable {variable value} {
		variable reverse_run_steps		;# Int: Number of steps which can be taken back
		variable ignore_watchdog_reset		;# Bool: Ignore reset invoked by watchdog overflow
		variable ignore_read_from_wr_only	;# Bool: Ignore reading from read only register
		variable ignore_invalid_reti		;# Bool: Ignore invalid return fom interrupt
		variable ignore_stack_overflow		;# Bool: Do not show "Stack overflow" dialog
		variable ignore_stack_underflow		;# Bool: Do not show "Stack underflow" dialog
		variable ignore_invalid_ins		;# Bool: Ignore invalid instructions
		variable ignore_invalid_IDATA		;# Bool: Ignore access to unimplemented IDATA memory
		variable ignore_invalid_EDATA		;# Bool: Ignore access to unimplemented EDATA memory
		variable ignore_invalid_XDATA		;# Bool: Ignore access to unimplemented XDATA memory
		variable ignore_invalid_BIT		;# Bool: Ignore access to unimplemented bit
		variable ignore_invalid_CODE		;# Bool: Ignore access to unimplemented CODE memory
		variable ignore_EEPROM_WR_fail		;# Bool: Ignore EEPROM write failure
		variable ignore_EEPROM_WR_abort		;# Bool: Ignore EEPROM write abort
		variable undefined_value		;# Int: How to handle undefined values (0 == 0; 1 == 255; 2 == random)

		getSettings
		set $variable $value
		use_settings
		save_config
	}

	## Retrieve settings from simulator NS
	 # @return void
	proc getSettings {} {
		variable defaults		;# List of default settings

		# Set local option variables
		foreach var $defaults {
			set var [lindex $var 0]
			set ::configDialogs::simulator::${var} [subst "\$::Simulator::$var"]
		}
	}

	## Set simulator acording to local settings
	 # @return void
	proc use_settings {} {
		variable reverse_run_steps	;# Int: Number of steps which can be taken back
		variable defaults		;# List of default settings

		# Adjust RR stack capacity
		if {$reverse_run_steps == {}} {
			set reverse_run_steps 0
		}

		# Set option variables
		foreach var $defaults {
			set var [lindex $var 0]
			set ::Simulator::$var [subst "\$::configDialogs::simulator::${var}"]
		}
	}

	## Save settings to the config file
	 # @return void
	proc save_config {} {
		variable defaults	;# List of default settings

		# Save option variables
		foreach var $defaults {
			set var [lindex $var 0]
			::settings setValue "Simulator/$var" [subst "\$::Simulator::${var}"]
		}

		# Synchronize
		::settings saveConfig
	}

	## Load settings from config file
	 # @return void
	proc load_config {} {
		variable defaults	;# List of default settings

		# Load normal options
		foreach item $defaults {
			set var [lindex $item 0]
			set val [lindex $item 1]
			set ::Simulator::${var} [::settings getValue "Simulator/$var" $val]
		}
	}

	## Destroy the dialog
	 # @return void
	proc CANCEL {} {
		variable win		;# ID of toplevel dialog window
		variable dialog_opened	;# Bool: True if this dialog is already opened

		# Destroy dialog window
		set dialog_opened 0
		grab release $win
		destroy $win
	}

	## Use settings and destroy the dialog
	 # @return void
	proc OK {} {
		# Use and save settings
		use_settings
		save_config

		# Destroy dialog window
		CANCEL
	}

	## Restrore defaults
	 # @return void
	proc DEFAULTS {} {
		variable win		;# ID of toplevel dialog window
		variable defaults	;# List of default settings

		if {[tk_messageBox		\
			-parent $win		\
			-type yesno		\
			-icon question		\
			-title [mc "Are you sure ?"]	\
			-message [mc "Are you sure you want to resore default settings"]	\
		] != {yes}} {
			return
		}

		# Restore normal options
		foreach item $defaults {
			set var [lindex $item 0]
			set val [lindex $item 1]
			set ::configDialogs::simulator::${var} $val
		}
	}
}
