/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Murray Cumming <murrayc@openismus.com>
 */

#include <grilo.h>
#include <stdio.h>

#define EXAMPLE_GRL_MEDIA_SCANNER_PLUGIN_ID "grl-mediascanner"

GMainLoop* loop = NULL;

static void
browse_cb (GrlSource *source G_GNUC_UNUSED,
           guint browse_id G_GNUC_UNUSED,
           GrlMedia *media,
           guint remaining,
           gpointer user_data G_GNUC_UNUSED,
           const GError *error)
{
    /* First we check if the operation failed for some reason */
    g_assert_no_error (error);

    if (media) {
        const gchar *title = grl_media_get_title (media);

        /** The grl-mediascanner plugin does not provide a hierarchy,
         * but let's check for a container anyway.
         */
        if (GRL_IS_MEDIA_BOX (media)) {
             guint childcount = grl_media_box_get_childcount (GRL_MEDIA_BOX (media));
             printf ("Container: title='%s', child count=%d\n", title, childcount);
         } else {
             guint seconds = grl_media_get_duration (media);
             const gchar *url = grl_media_get_url (media);
             printf ("Media: title='%s', length(seconds)=%d\n", title, seconds);
             printf ("  URL=%s\n", url);
         }

         g_object_unref (media);
    }

    /** Stop the main loop when we are finished.
     */
    if (remaining <= 0) {
        g_main_loop_quit (loop);
    }
}

int main(int argc, char *argv[]) {

    /*
     * These defines are set by the build system.
     * Uncomment this to use the installed grilo plugins,
     * instead of the locally-built grilo plugins.
     */
    g_setenv (GRL_PLUGIN_PATH_VAR, EXAMPLE_GRILO_PLUGIN_DIR, TRUE);
    grl_init (&argc, &argv);

    /*
     * Load the Grilo plugin:
     */
    GrlRegistry *registry = grl_registry_get_default();

    GError *error = NULL;
    gboolean plugin_loaded = grl_registry_load_plugin_by_id (
        registry, EXAMPLE_GRL_MEDIA_SCANNER_PLUGIN_ID, &error);
    g_assert (plugin_loaded);
    g_assert_no_error (error);

    /*
     * Get the plugin:
     *
    GrlPlugin *plugin = 
        grl_registry_lookup_plugin (registry, 
            EXAMPLE_GRL_MEDIA_SCANNER_PLUGIN_ID);
    g_assert (plugin);
     */


    /**
     * Get the Grilo Source:
     */
    GrlSource *source =
        grl_registry_lookup_source(registry, EXAMPLE_GRL_MEDIA_SCANNER_PLUGIN_ID);
    g_assert (source);


    /*
     * List the content:
     */
    g_assert (grl_source_supported_operations (source) & GRL_OP_QUERY);
    GrlCaps *caps = grl_source_get_caps (source, GRL_OP_QUERY);
    g_assert (caps);

    GrlOperationOptions *options = grl_operation_options_new (caps);
    grl_operation_options_set_count (options, 5);
    grl_operation_options_set_flags (options, GRL_RESOLVE_IDLE_RELAY);

    GList * keys = grl_metadata_key_list_new (GRL_METADATA_KEY_TITLE,
        GRL_METADATA_KEY_DURATION,
        GRL_METADATA_KEY_URL,
        GRL_METADATA_KEY_CHILDCOUNT,
        NULL);

    /*
     * The query string syntax is plugin-specific.
     * TODO(M4): This uses the lucene query syntax: https://lucene.apache.org/core/3_6_0/queryparsersyntax.html
     * TODO(M4): Link to our gtk-doc docs for our grilo plugin.
     */
    grl_source_query (source,
                       "title:Soul*",
                       keys,
                       options,
                       browse_cb,
                       NULL);

    /*
     * Start the main loop so our callback can be called:
     */
    loop = g_main_loop_new (NULL, FALSE);
    g_main_loop_run (loop);

    /*
     * Release objects:
     */
    g_list_free (keys);
    g_object_unref (caps);
    g_object_unref (options);

    return 0;
}
