#include <Python.h>
#include <stdlib.h>
#include <math.h>
#include <complex.h>	// Use native C99 complex type for fftw3
#include "quisk.h"
#include "filter.h"

static enum {
	Unknown,
	None,
	Four,
	Five,
	Eight,
	FiveFour,
	FiveEight,
	FourEight,
	ThreeEight,
	TwoEight
} idecim_scheme;

// Decimation filters are Parks-McClellan FIR Filter Design:
// Order: 81
// Passband ripple: 0.1 dB
// Transition band: 0.05
// Stopband attenuation: 100.0 dB

#define DEC_FILT_TAPS		82

static double dec_filt_four[DEC_FILT_TAPS] = {	// 0.200 bandwidth for 1/4
6.93262847909316E-5, 1.6197926963904747E-4, -2.978880216503281E-5, -7.607072256017523E-4,
-0.0015328465141652313, -0.0012468590175053562, 2.610559866853963E-4, 0.0012944734487453414,
2.3302486450800912E-4, -0.0016407424838786528, -0.0012416553276332963, 0.0015073572516228356,
0.00246088370832881, -7.03592657698059E-4, -0.003615088283354994, -9.287652193144898E-4,
0.004243002194452015, 0.003314623044307713, -0.003839298846344248, -0.006115837049852382,
0.0019601369874323576, 0.008710630191084028, 0.0016299333322581358, -0.010249913305703927,
-0.006832131686428464, 0.009766175058001373, 0.013128572492857463, -0.006308334625132938,
-0.019542504322817577, -9.249612897081307E-4, 0.024651789635613427, 0.012545482681757834,
-0.026576155055309945, -0.029184569947365617, 0.022677733544428112, 0.0524099453299258,
-0.007945906430265487, -0.08887991706016465, -0.03541624039036922, 0.18706014148382252,
0.4027578624778919, 0.4027578624778919, 0.18706014148382252, -0.03541624039036922,
-0.08887991706016465, -0.007945906430265487, 0.0524099453299258, 0.022677733544428112,
-0.029184569947365617, -0.026576155055309945, 0.012545482681757834, 0.024651789635613427,
-9.249612897081307E-4, -0.019542504322817577, -0.006308334625132938, 0.013128572492857463,
0.009766175058001373, -0.006832131686428464, -0.010249913305703927, 0.0016299333322581358,
0.008710630191084028, 0.0019601369874323576, -0.006115837049852382, -0.003839298846344248,
0.003314623044307713, 0.004243002194452015, -9.287652193144898E-4, -0.003615088283354994,
-7.03592657698059E-4, 0.00246088370832881, 0.0015073572516228356, -0.0012416553276332963,
-0.0016407424838786528, 2.3302486450800912E-4, 0.0012944734487453414, 2.610559866853963E-4,
-0.0012468590175053562, -0.0015328465141652313, -7.607072256017523E-4, -2.978880216503281E-5,
1.6197926963904747E-4, 6.93262847909316E-5};

static double dec_filt_five[DEC_FILT_TAPS] = {	// 0.150 bandwidth for 1/5
4.802945694175824E-5, 1.3678129104226564E-4, 1.5780237068504314E-4, -9.447967308879232E-5,
-7.096795320832177E-4, -0.001401250667455834, -0.0015400533718900225, -6.747045776483936E-4,
8.153280522959677E-4, 0.0016993691656680282, 8.936602390940594E-4, -0.0012352048814478292,
-0.002710208527872082, -0.001616712517641334, 0.0016582434482831638, 0.004079060014727778,
0.0026053079020420494, -0.0022732852688599737, -0.0059919633490814915, -0.003956217667702122,
0.003149204678409682, 0.008619113656227065, 0.00575247818483739, -0.004432293133200357,
-0.012256528072905008, -0.008148410734226686, 0.006362564035902175, 0.01743705847413967,
0.011452631286430849, -0.00939474436497328, -0.025278755048946437, -0.016382944751566725,
0.014588955459961006, 0.03869591981083795, 0.02505025857670139, -0.025262163858126217,
-0.0684261501517449, -0.04695060459326292, 0.060595955694417517, 0.21142646204362847,
0.32063462646029334, 0.32063462646029334, 0.21142646204362847, 0.060595955694417517,
-0.04695060459326292, -0.0684261501517449, -0.025262163858126217, 0.02505025857670139,
0.03869591981083795, 0.014588955459961006, -0.016382944751566725, -0.025278755048946437,
-0.00939474436497328, 0.011452631286430849, 0.01743705847413967, 0.006362564035902175,
-0.008148410734226686, -0.012256528072905008, -0.004432293133200357, 0.00575247818483739,
0.008619113656227065, 0.003149204678409682, -0.003956217667702122, -0.0059919633490814915,
-0.0022732852688599737, 0.0026053079020420494, 0.004079060014727778, 0.0016582434482831638,
-0.001616712517641334, -0.002710208527872082, -0.0012352048814478292, 8.936602390940594E-4,
0.0016993691656680282, 8.153280522959677E-4, -6.747045776483936E-4, -0.0015400533718900225,
-0.001401250667455834, -7.096795320832177E-4, -9.447967308879232E-5, 1.5780237068504314E-4,
1.3678129104226564E-4, 4.802945694175824E-5};

static double dec_filt_eight[DEC_FILT_TAPS] = {	// 0.075 bandwidth for 1/8
3.189499939137948E-5, 9.658000074768447E-5, 2.066544953451926E-4, 3.44452671428755E-4,
4.5985201258950855E-4, 4.7151300219738324E-4, 2.884095680130089E-4, -1.4972784756879228E-4,
-8.240738514407132E-4, -0.001601285443271075, -0.0022351411796416518, -0.0024218067343669786,
-0.0019063148533499982, -6.12297195964114E-4, 0.0012546716666281578, 0.0031864584051393505,
0.004473200480571965, 0.004424166162645936, 0.0026536808739181, -6.675516541696363E-4,
-0.004711222046697417, -0.00812616121435781, -0.00942114518562818, -0.007510029585918773,
-0.0022454680141598088, 0.00527145304120975, 0.012781816045613894, 0.017422645491741433,
0.016645094307362603, 0.009241600363100413, -0.0038641474640177136, -0.019390756369107805,
-0.032221251420502695, -0.036629981372148956, -0.02792435737553998, -0.004033420538463705,
0.03346490593697525, 0.07926024983841577, 0.12528860198355965, 0.16256947281001224,
0.18341920540549964, 0.18341920540549964, 0.16256947281001224, 0.12528860198355965,
0.07926024983841577, 0.03346490593697525, -0.004033420538463705, -0.02792435737553998,
-0.036629981372148956, -0.032221251420502695, -0.019390756369107805, -0.0038641474640177136,
0.009241600363100413, 0.016645094307362603, 0.017422645491741433, 0.012781816045613894,
0.00527145304120975, -0.0022454680141598088, -0.007510029585918773, -0.00942114518562818,
-0.00812616121435781, -0.004711222046697417, -6.675516541696363E-4, 0.0026536808739181,
0.004424166162645936, 0.004473200480571965, 0.0031864584051393505, 0.0012546716666281578,
-6.12297195964114E-4, -0.0019063148533499982, -0.0024218067343669786, -0.0022351411796416518,
-0.001601285443271075, -8.240738514407132E-4, -1.4972784756879228E-4, 2.884095680130089E-4,
4.7151300219738324E-4, 4.5985201258950855E-4, 3.44452671428755E-4, 2.066544953451926E-4,
9.658000074768447E-5, 3.189499939137948E-5};

double quiskMicFilt48Coefs[325] = {
 0.000070000950338734, 0.000060542525732639, 0.000084830915167248, 0.000113820318991206, 0.000147523658260692, 
 0.000185727097247136, 0.000228017694264326, 0.000273714746744222, 0.000321902242967596, 0.000371355069437399, 
 0.000420595492180569, 0.000467865412686539, 0.000511235564174107, 0.000548584697295435, 0.000577683697404093, 
 0.000596168573595124, 0.000601725499419238, 0.000592142083607333, 0.000565459097743531, 0.000519884471131262, 
 0.000454031497680723, 0.000367015380281216, 0.000258599355817496, 0.000128888056076112, -0.000021074826181109, 
-0.000189585802186493, -0.000374389987374961, -0.000572207780852182, -0.000779359325040342, -0.000991335380833019, 
-0.001203244280115714, -0.001409670281637694, -0.001605003575053789, -0.001783471817219780, -0.001939438753998784, 
-0.002067470485014850, -0.002162628068669760, -0.002220597743425130, -0.002237960051673301, -0.002212264773625852, 
-0.002142247186138010, -0.002027907322210780, -0.001870648667784210, -0.001673216059663119, -0.001439782952032160, 
-0.001175849407968695, -0.000888138367181945, -0.000584400564491252, -0.000273350274681290, 0.000035800336978808, 
 0.000333410168961052, 0.000609868204837889, 0.000855845198751289, 0.001062645743133108, 0.001222567778915907, 
 0.001329178561171123, 0.001377619146269160, 0.001364836763234778, 0.001289791422827911, 0.001153569049793089, 
 0.000959442606127223, 0.000712858984801055, 0.000421343499696318, 0.000094311703309987, -0.000257178812726900, 
-0.000620740024523200, -0.000983067401076951, -0.001330378208545664, -0.001648868642029554, -0.001925231739749957, 
-0.002147185427808426, -0.002303924955683100, -0.002386617358290471, -0.002388809260074332, -0.002306735130576522, 
-0.002139640086196955, -0.001889862283127962, -0.001562946712603551, -0.001167542853751166, -0.000715240442693352, 
-0.000220259485535331, 0.000300952307259904, 0.000830234179165372, 0.001348316435464076, 0.001835484518420602, 
 0.002272286892056382, 0.002640275270224598, 0.002922747186895021, 0.003105429025669433, 0.003177121318055631, 
 0.003130264761140124, 0.002961377569599093, 0.002671367231884301, 0.002265711108816848, 0.001754448044185524, 
 0.001152016769339380, 0.000476936676175902, -0.000248718444031171, -0.000999948124398612, -0.001749629130166096, 
-0.002469433330814752, -0.003130790556013717, -0.003705967375692800, -0.004169105103972317, -0.004497290920890501, 
-0.004671516289830527, -0.004677586380416906, -0.004506861602367489, -0.004156859083959776, -0.003631621917386205, 
-0.002941916481868308, -0.002105153002563826, -0.001145089028022731, -0.000091277027194512, 0.001021707212306502, 
 0.002155280663061357, 0.003268024493993296, 0.004317048090841251, 0.005259445594087794, 0.006053837753595639, 
 0.006661931665353417, 0.007050060676035087, 0.007190601958065676, 0.007063321534553406, 0.006656461303944376, 
 0.005967650471713226, 0.005004501271125813, 0.003784934033421286, 0.002337151835923401, 0.000699290786282728, 
-0.001081281658450751, -0.002948980083931280, -0.004841350286243770, -0.006690670765283031, -0.008425801537676034, 
-0.009974215633219711, -0.011264158097802190, -0.012226858155753686, -0.012798749185527288, -0.012923592591893964, 
-0.012554478278462186, -0.011655599268700838, -0.010203756484326235, -0.008189534541107277, -0.005618124746137401, 
-0.002509708641553183, 0.001100549166042612, 0.005162968475167027, 0.009614243563845803, 0.014378746109223314, 
 0.019370216567322070, 0.024493786616406266, 0.029648291853908371, 0.034728809072115958, 0.039629330964538544, 
 0.044245520769433347, 0.048477462256357018, 0.052232311099492307, 0.055426788736225432, 0.057989428195401226, 
 0.059862511141804443, 0.061003649799348830, 0.061386934931288717, 0.061003649799348830, 0.059862511141804443, 
 0.057989428195401226, 0.055426788736225432, 0.052232311099492307, 0.048477462256357018, 0.044245520769433347, 
 0.039629330964538544, 0.034728809072115958, 0.029648291853908371, 0.024493786616406266, 0.019370216567322070, 
 0.014378746109223314, 0.009614243563845803, 0.005162968475167027, 0.001100549166042612, -0.002509708641553183, 
-0.005618124746137401, -0.008189534541107277, -0.010203756484326235, -0.011655599268700838, -0.012554478278462186, 
-0.012923592591893964, -0.012798749185527288, -0.012226858155753686, -0.011264158097802190, -0.009974215633219711, 
-0.008425801537676034, -0.006690670765283031, -0.004841350286243770, -0.002948980083931280, -0.001081281658450751, 
 0.000699290786282728, 0.002337151835923401, 0.003784934033421286, 0.005004501271125813, 0.005967650471713226, 
 0.006656461303944376, 0.007063321534553406, 0.007190601958065676, 0.007050060676035087, 0.006661931665353417, 
 0.006053837753595639, 0.005259445594087794, 0.004317048090841251, 0.003268024493993296, 0.002155280663061357, 
 0.001021707212306502, -0.000091277027194512, -0.001145089028022731, -0.002105153002563826, -0.002941916481868308, 
-0.003631621917386205, -0.004156859083959776, -0.004506861602367489, -0.004677586380416906, -0.004671516289830527, 
-0.004497290920890501, -0.004169105103972317, -0.003705967375692800, -0.003130790556013717, -0.002469433330814752, 
-0.001749629130166096, -0.000999948124398612, -0.000248718444031171, 0.000476936676175902, 0.001152016769339380, 
 0.001754448044185524, 0.002265711108816848, 0.002671367231884301, 0.002961377569599093, 0.003130264761140124, 
 0.003177121318055631, 0.003105429025669433, 0.002922747186895021, 0.002640275270224598, 0.002272286892056382, 
 0.001835484518420602, 0.001348316435464076, 0.000830234179165372, 0.000300952307259904, -0.000220259485535331, 
-0.000715240442693352, -0.001167542853751166, -0.001562946712603551, -0.001889862283127962, -0.002139640086196955, 
-0.002306735130576522, -0.002388809260074332, -0.002386617358290471, -0.002303924955683100, -0.002147185427808426, 
-0.001925231739749957, -0.001648868642029554, -0.001330378208545664, -0.000983067401076951, -0.000620740024523200, 
-0.000257178812726900, 0.000094311703309987, 0.000421343499696318, 0.000712858984801055, 0.000959442606127223, 
 0.001153569049793089, 0.001289791422827911, 0.001364836763234778, 0.001377619146269160, 0.001329178561171123, 
 0.001222567778915907, 0.001062645743133108, 0.000855845198751289, 0.000609868204837889, 0.000333410168961052, 
 0.000035800336978808, -0.000273350274681290, -0.000584400564491252, -0.000888138367181945, -0.001175849407968695, 
-0.001439782952032160, -0.001673216059663119, -0.001870648667784210, -0.002027907322210780, -0.002142247186138010, 
-0.002212264773625852, -0.002237960051673301, -0.002220597743425130, -0.002162628068669760, -0.002067470485014850, 
-0.001939438753998784, -0.001783471817219780, -0.001605003575053789, -0.001409670281637694, -0.001203244280115714, 
-0.000991335380833019, -0.000779359325040342, -0.000572207780852182, -0.000374389987374961, -0.000189585802186493, 
-0.000021074826181109, 0.000128888056076112, 0.000258599355817496, 0.000367015380281216, 0.000454031497680723, 
 0.000519884471131262, 0.000565459097743531, 0.000592142083607333, 0.000601725499419238, 0.000596168573595124, 
 0.000577683697404093, 0.000548584697295435, 0.000511235564174107, 0.000467865412686539, 0.000420595492180569, 
 0.000371355069437399, 0.000321902242967596, 0.000273714746744222, 0.000228017694264326, 0.000185727097247136, 
 0.000147523658260692, 0.000113820318991206, 0.000084830915167248, 0.000060542525732639, 0.000070000950338734} ;

// Rate 8000, pass 1350, stop 1700, ripple 0.2 dB, Atten 100 dB, taps 93 (tune 1650)
double quiskMicFilt8Coefs[93] = { -0.000048361334397216, -0.000162172832480051, -0.000233937595698212,
-0.000006327103457766, 0.000686150429731403, 0.001555572669911020, 0.001852235322638049, 0.001010518676959113,
-0.000533703037637750, -0.001395527281294679, -0.000547805689731835, 0.001291819787342756, 0.002005877810728077,
 0.000349773034952121, -0.002188174349636707, -0.002546652809598079, 0.000326796204088075, 0.003522458156729808,
 0.002929815624806958, -0.001643263026946036, -0.005188443834846179, -0.002794087489903680, 0.003828812643532208,
 0.006979098861533383, 0.001700757602284474, -0.007013801827881641, -0.008485813922195549, 0.000867966129209818,
 0.011177153889901327, 0.009069558799223736, -0.005488279673121778, -0.016106658511052186, -0.007822070151870978,
 0.012835116756370880, 0.021388877055372072, 0.003393318192541889, -0.023987911353715568, -0.026461891098409809,
 0.006724804206592470, 0.041707244257452522, 0.030690541453250592, -0.029955120082834141, -0.077739106243522968,
-0.033499374091835024, 0.116943716609774510, 0.290907261177841880, 0.367819184749133500, 0.290907261177841880,
 0.116943716609774510, -0.033499374091835024, -0.077739106243522968, -0.029955120082834141, 0.030690541453250592,
 0.041707244257452522, 0.006724804206592470, -0.026461891098409809, -0.023987911353715568, 0.003393318192541889,
 0.021388877055372072, 0.012835116756370880, -0.007822070151870978, -0.016106658511052186, -0.005488279673121778,
 0.009069558799223736, 0.011177153889901327, 0.000867966129209818, -0.008485813922195549, -0.007013801827881641,
 0.001700757602284474, 0.006979098861533383, 0.003828812643532208, -0.002794087489903680, -0.005188443834846179,
-0.001643263026946036, 0.002929815624806958, 0.003522458156729808, 0.000326796204088075, -0.002546652809598079,
-0.002188174349636707, 0.000349773034952121, 0.002005877810728077, 0.001291819787342756, -0.000547805689731835,
-0.001395527281294679, -0.000533703037637750, 0.001010518676959113, 0.001852235322638049, 0.001555572669911020,
 0.000686150429731403, -0.000006327103457766, -0.000233937595698212, -0.000162172832480051, -0.000048361334397216} ;

// Rate 48000, pass 3200, stop 4500, ripple 0.2 dB, atten 100 dB, taps 144
double quiskLpFilt48Coefs[144] = {-0.000017125474330604, -0.000040011820393130, -0.000077862936583202,
-0.000127389313997705, -0.000181102741386044, -0.000225109758175834, -0.000239838685218700, -0.000202969232396477,
-0.000094305597311178, 0.000097794345082862, 0.000369998664433527, 0.000699241419612492, 0.001041511593601482,
 0.001335775967586111, 0.001513363005661080, 0.001512024052854210, 0.001292282575245647, 0.000852402910741175,
 0.000237845095399279, -0.000458531344106017, -0.001107710877113322, -0.001566990390963083, -0.001710431193735069,
-0.001461394216561484, -0.000819564995596107, 0.000125333340929399, 0.001199019317168316, 0.002169473500980469,
 0.002792319280177214, 0.002866870662163649, 0.002290930866878644, 0.001100704907673480, -0.000516328530790399,
-0.002243035863158312, -0.003688859468654476, -0.004472285396379554, -0.004313484308907802, -0.003116141834256676,
-0.001017067737117166, 0.001613055660814351, 0.004224931456170525, 0.006196508693202270, 0.006971075945479586,
 0.006198930990039089, 0.003849274053846188, 0.000261650017776247, -0.003883392014577992, -0.007678244207599024,
-0.010171159926985415, -0.010584341389004935, -0.008520924878120863, -0.004111405600230054, 0.001942089793582858,
 0.008446987602879255, 0.013924611888810799, 0.016907513230850324, 0.016275537240591227, 0.011561878706483661,
 0.003159517846795821, -0.007627342018415888, -0.018714786943998040, -0.027545855699856577, -0.031527533359270804,
-0.028522764468194239, -0.017305127467535198, 0.002115380882896933, 0.028367517836585164, 0.058831871715231798,
 0.089984540025951881, 0.117919393961330910, 0.138960802368858470, 0.150258729900921100, 0.150258729900921100,
 0.138960802368858470, 0.117919393961330910, 0.089984540025951881, 0.058831871715231798, 0.028367517836585164,
 0.002115380882896933, -0.017305127467535198, -0.028522764468194239, -0.031527533359270804, -0.027545855699856577,
-0.018714786943998040, -0.007627342018415888, 0.003159517846795821, 0.011561878706483661, 0.016275537240591227,
 0.016907513230850324, 0.013924611888810799, 0.008446987602879255, 0.001942089793582858, -0.004111405600230054,
-0.008520924878120863, -0.010584341389004935, -0.010171159926985415, -0.007678244207599024, -0.003883392014577992,
 0.000261650017776247, 0.003849274053846188, 0.006198930990039089, 0.006971075945479586, 0.006196508693202270,
 0.004224931456170525, 0.001613055660814351, -0.001017067737117166, -0.003116141834256676, -0.004313484308907802,
-0.004472285396379554, -0.003688859468654476, -0.002243035863158312, -0.000516328530790399, 0.001100704907673480,
 0.002290930866878644, 0.002866870662163649, 0.002792319280177214, 0.002169473500980469, 0.001199019317168316,
 0.000125333340929399, -0.000819564995596107, -0.001461394216561484, -0.001710431193735069, -0.001566990390963083,
-0.001107710877113322, -0.000458531344106017, 0.000237845095399279, 0.000852402910741175, 0.001292282575245647,
 0.001512024052854210, 0.001513363005661080, 0.001335775967586111, 0.001041511593601482, 0.000699241419612492,
 0.000369998664433527, 0.000097794345082862, -0.000094305597311178, -0.000202969232396477, -0.000239838685218700,
-0.000225109758175834, -0.000181102741386044, -0.000127389313997705, -0.000077862936583202, -0.000040011820393130,
-0.000017125474330604 } ;

// Sample 1 Hz, pass 0.125, stop 0.192, ripple 0.1 dB, atten 100 dB, taps 64
double quiskFilt12_19Coefs[64] = { 0.000043810351462319, 0.000138863869398690, 0.000206835545025070,
 0.000047730531591317, -0.000514727559529336, -0.001353635480056038, -0.001881242409327435, -0.001336759882352925,
 0.000480762137654505, 0.002595795305500315, 0.003157538223935716, 0.000883738175457450, -0.003304786290465149,
-0.006137600456523736, -0.004232173904239833, 0.002529546544381186, 0.009496143474996962, 0.009949625572871518,
 0.000962602913419665, -0.012260655539427043, -0.018443416827456323, -0.008986096201165510, 0.012737225552167929,
 0.030249086308101754, 0.024812387650292519, -0.007810917817793795, -0.047781081559953899, -0.059071807532626572,
-0.012800932712041477, 0.089119808681493592, 0.208124268210933110, 0.287998295491409540, 0.287998295491409540,
 0.208124268210933110, 0.089119808681493592, -0.012800932712041477, -0.059071807532626572, -0.047781081559953899,
-0.007810917817793795, 0.024812387650292519, 0.030249086308101754, 0.012737225552167929, -0.008986096201165510,
-0.018443416827456323, -0.012260655539427043, 0.000962602913419665, 0.009949625572871518, 0.009496143474996962,
 0.002529546544381186, -0.004232173904239833, -0.006137600456523736, -0.003304786290465149, 0.000883738175457450,
 0.003157538223935716, 0.002595795305500315, 0.000480762137654505, -0.001336759882352925, -0.001881242409327435,
-0.001353635480056038, -0.000514727559529336, 0.000047730531591317, 0.000206835545025070, 0.000138863869398690,
 0.000043810351462319} ;

static int iDecimateFour(complex * cSamples, int nSamples, int idecim)
{  // Decimate to a lower sample rate by an integer idecim 2 through 4
	int i, j, k, n;
	complex cx;
	static int findex = 0;
	static complex bufDecim[MAX_FILTER_SIZE];
	static int counter = 0;

	n = 0;
	for (i = 0; i < nSamples; i++) {
		bufDecim[findex] = cSamples[i];
		if (++counter >= idecim) {
			counter = 0;		// output a sample
			cx = 0;
			j = findex;
			for (k = 0; k < DEC_FILT_TAPS; k++) {
				cx += bufDecim[j] * dec_filt_four[k];
				if (++j >= DEC_FILT_TAPS)
					j = 0;
			}
			cSamples[n++] = cx;
		}
		if (++findex >= DEC_FILT_TAPS)
			findex = 0;
	}
	return n;
}

static int iDecimateFive(complex * cSamples, int nSamples, int idecim)
{  // Decimate to a lower sample rate by an integer idecim 2 through 5
	int i, j, k, n;
	complex cx;
	static int findex = 0;
	static complex bufDecim[MAX_FILTER_SIZE];
	static int counter = 0;

	n = 0;
	for (i = 0; i < nSamples; i++) {
		bufDecim[findex] = cSamples[i];
		if (++counter >= idecim) {
			counter = 0;		// output a sample
			cx = 0;
			j = findex;
			for (k = 0; k < DEC_FILT_TAPS; k++) {
				cx += bufDecim[j] * dec_filt_five[k];
				if (++j >= DEC_FILT_TAPS)
					j = 0;
			}
			cSamples[n++] = cx;
		}
		if (++findex >= DEC_FILT_TAPS)
			findex = 0;
	}
	return n;
}

static int iDecimateEight(complex * cSamples, int nSamples, int idecim)
{  // Decimate to a lower sample rate by an integer idecim 2 through 8
	int i, j, k, n;
	complex cx;
	static int findex = 0;
	static complex bufDecim[MAX_FILTER_SIZE];
	static int counter = 0;

	n = 0;
	for (i = 0; i < nSamples; i++) {
		bufDecim[findex] = cSamples[i];
		if (++counter >= idecim) {
			counter = 0;		// output a sample
			cx = 0;
			j = findex;
			for (k = 0; k < DEC_FILT_TAPS; k++) {
				cx += bufDecim[j] * dec_filt_eight[k];
				if (++j >= DEC_FILT_TAPS)
					j = 0;
			}
			cSamples[n++] = cx;
		}
		if (++findex >= DEC_FILT_TAPS)
			findex = 0;
	}
	return n;
}

// Decimate: Lower the sample rate by idecim.  This function uses
// static storage, and there must be only one call to it in the program.
// Using two calls to decimate two sample streams will not work.
// Check the resulting bandwidth for your decimation, as it will
// generally be much less than the maximum.
int quisk_iDecimate(complex * cSamples, int nSamples, int idecim)
{
	static int old_idecim = 0;		// previous value of idecim

	if (idecim != old_idecim) {		// Initialization
		old_idecim = idecim;
		if (idecim <= 1)	// Set the correct decimation filter based on rate reduction.
			idecim_scheme = None;
		else if (idecim <= 4)
			idecim_scheme = Four;
		else if (idecim == 5)
			idecim_scheme = Five;
		else if (idecim <= 8)
			idecim_scheme = Eight;
		else if (idecim % 5 == 0) {		// good for 10, 15, ..., 40
			if (idecim / 5 <= 4)
				idecim_scheme = FiveFour;
			else
				idecim_scheme = FiveEight;
		}
		else if (idecim % 4 == 0)
			idecim_scheme = FourEight;
		else if (idecim % 3 == 0)
			idecim_scheme = ThreeEight;
		else if (idecim % 2 == 0)
			idecim_scheme = TwoEight;
		else		// There is no good scheme, and this filter is inadequate
			idecim_scheme = Eight;
	}
	switch (idecim_scheme) {
		case None:
			break;
		case Four:
			nSamples = iDecimateFour(cSamples, nSamples, idecim);
			break;
		case Five:
			nSamples = iDecimateFive(cSamples, nSamples, idecim);
			break;
		case Eight:
			nSamples = iDecimateEight(cSamples, nSamples, idecim);
			break;
		case FiveFour:
			nSamples = iDecimateFive(cSamples, nSamples, 5);
			nSamples = iDecimateFour(cSamples, nSamples, idecim / 5);
			break;
		case FiveEight:
			nSamples = iDecimateFive(cSamples, nSamples, 5);
			nSamples = iDecimateEight(cSamples, nSamples, idecim / 5);
			break;
		case FourEight:
			nSamples = iDecimateFour(cSamples, nSamples, 4);
			nSamples = iDecimateEight(cSamples, nSamples, idecim / 4);
			break;
		case ThreeEight:
			nSamples = iDecimateFour(cSamples, nSamples, 3);
			nSamples = iDecimateEight(cSamples, nSamples, idecim / 3);
			break;
		case TwoEight:
			nSamples = iDecimateFour(cSamples, nSamples, 2);
			nSamples = iDecimateEight(cSamples, nSamples, idecim / 2);
			break;
		default:		// should not happen
			nSamples = iDecimateEight(cSamples, nSamples, idecim);
			break;
	}
	return nSamples;	// return the new number of samples
}

void quisk_filt_cInit(struct quisk_cFilter * filter, double * coefs, int taps)
{	// Prepare a new filter using coefs and taps.  Samples are complex.
	filter->dCoefs = coefs;
	filter->cpxCoefs = NULL;
	filter->cSamples = (complex *)malloc(taps * sizeof(complex));
	memset(filter->cSamples, 0, taps * sizeof(complex));
	filter->ptcSamp = filter->cSamples;
	filter->nTaps = taps;
	filter->counter = 0;
	filter->cBuf = NULL;
	filter->dBuf = NULL;
	filter->nBuf = 0;
}

void quisk_filt_dInit(struct quisk_dFilter * filter, double * coefs, int taps)
{	// Prepare a new filter using coefs and taps.  Samples are double.
	filter->dCoefs = coefs;
	filter->cpxCoefs = NULL;
	filter->dSamples = (double *)malloc(taps * sizeof(double));
	memset(filter->dSamples, 0, taps * sizeof(double));
	filter->ptdSamp = filter->dSamples;
	filter->nTaps = taps;
	filter->counter = 0;
	filter->cBuf = NULL;
	filter->dBuf = NULL;
	filter->nBuf = 0;
}
	

void quisk_filt_tune(struct quisk_dFilter * filter, double freq, int ssb_upper)
{	// Tune a filter into an analytic I/Q filter with complex coefficients.
	// freq is the center frequency / sample rate.  Reverse coef if ssb_upper == 0
	int i;
	complex coef, tune;
	double D;

	if ( ! filter->cpxCoefs)
		filter->cpxCoefs = (complex *)malloc(filter->nTaps * sizeof(complex));
	tune = I * 2.0 * M_PI * freq;
	D = (filter->nTaps - 1.0) / 2.0;
	for (i = 0; i < filter->nTaps; i++) {
		coef = 2.0 * cexp(tune * (i - D)) * filter->dCoefs[i];
		if (ssb_upper)
			filter->cpxCoefs[i] = coef;
		else
			filter->cpxCoefs[i] = cimag(coef) + I * creal(coef);
	}
}

complex quisk_dC_out(double sample, struct quisk_dFilter * filter)
{
	complex csample;
	complex * ptCoef;
	double  * ptSample;
	int k;

	// FIR bandpass filter; separate double sample into I and Q.
	// Put samples into buffer left to right.  Use samples right to left.
	ptSample = filter->ptdSamp;
	*ptSample = sample;
	ptCoef = filter->cpxCoefs;
	csample = 0;
	for (k = 0; k < filter->nTaps; k++, ptCoef++) {
		csample += *ptSample  *  *ptCoef;
		if (--ptSample < filter->dSamples)
			ptSample = filter->dSamples + filter->nTaps - 1;
	}
	if (++filter->ptdSamp >= filter->dSamples + filter->nTaps)
		filter->ptdSamp = filter->dSamples;
	return csample;
}

int quisk_cInterpolate(complex * cSamples, int count, struct quisk_cFilter * filter, int interp)
{	// This uses the double coefficients of filter (not the complex).
	int i, j, k, nOut;
	double * ptCoef;
	complex * ptSample;
	complex csample;

	if (count > filter->nBuf) {	// increase size of sample buffer
		filter->nBuf = count * 2;
		if (filter->cBuf)
			free(filter->cBuf);
		filter->cBuf = (complex *)malloc(filter->nBuf * sizeof(complex));
	}
	memcpy(filter->cBuf, cSamples, count * sizeof(complex));
	nOut = 0;
	for (i = 0; i < count; i++) {
		// Put samples into buffer left to right.  Use samples right to left.
		*filter->ptcSamp = filter->cBuf[i];
		for (j = 0; j < interp; j++) {
			ptSample = filter->ptcSamp;
			ptCoef = filter->dCoefs + j;
			csample = 0;
			for (k = 0; k < filter->nTaps / interp; k++, ptCoef += interp) {
				csample += *ptSample  *  *ptCoef * interp;
				if (--ptSample < filter->cSamples)
					ptSample = filter->cSamples + filter->nTaps - 1;
			}
			cSamples[nOut++] = csample;
		}
		if (++filter->ptcSamp >= filter->cSamples + filter->nTaps)
			filter->ptcSamp = filter->cSamples;
	}
	return nOut;
}

int quisk_cDecimate(complex * cSamples, int count, struct quisk_cFilter * filter, int decim)
{	// This uses the double coefficients of filter (not the complex).
	int i, k, nOut;
	complex * ptSample;
	double * ptCoef;
	complex csample;

	nOut = 0;
	for (i = 0; i < count; i++) {
		*filter->ptcSamp = cSamples[i];
		if (++filter->counter >= decim) {
			filter->counter = 0;		// output a sample
			csample = 0;
			ptSample = filter->ptcSamp;
			ptCoef = filter->dCoefs;
			for (k = 0; k < filter->nTaps; k++, ptCoef++) {
				csample += *ptSample  *  *ptCoef;
				if (--ptSample < filter->cSamples)
					ptSample = filter->cSamples + filter->nTaps - 1;
			}
			cSamples[nOut++] = csample;
		}
		if (++filter->ptcSamp >= filter->cSamples + filter->nTaps)
			filter->ptcSamp = filter->cSamples;
	}
	return nOut;
}
	
